// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use serde::{Serialize, Deserialize};

#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(rename = "Player", deny_unknown_fields)]
pub struct PlayerConfig {
	#[serde(default = "default_speed")] pub speed: f32,
	#[serde(default = "default_max_health")] pub max_health: f32,
	#[serde(default = "default_regen_rate")] pub regen_rate: f32,
	#[serde(default = "default_regen_time")] pub regen_time: f32,
	#[serde(default)] pub bullets: BulletConfig,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(rename = "Bullet", deny_unknown_fields)]
pub struct BulletConfig {
	#[serde(default = "default_bullet_reload_interval")] pub reload_interval: f32,
	#[serde(default = "default_bullet_damage")] pub damage: f32,
	#[serde(default = "default_bullet_speed")] pub speed: f32,
	#[serde(default = "default_bullet_spread")] pub spread: f32,
	#[serde(default = "default_bullet_lifetime")] pub lifetime: f32,
}

impl Default for PlayerConfig {
	fn default() -> PlayerConfig {
		PlayerConfig {
			speed: default_speed(),
			max_health: default_max_health(),
			regen_rate: default_regen_rate(),
			regen_time: default_regen_time(),
			bullets: BulletConfig::default(),
		}
	}
}

impl Default for BulletConfig {
	fn default() -> BulletConfig {
		BulletConfig {
			reload_interval: default_bullet_reload_interval(),
			damage: default_bullet_damage(),
			speed: default_bullet_speed(),
			spread: default_bullet_spread(),
			lifetime: default_bullet_lifetime(),
		}
	}
}

fn default_speed() -> f32 { 12.0 }
fn default_max_health() -> f32 { MAX_HEALTH_DEFAULT }
fn default_regen_rate() -> f32 { 2.0 }
fn default_regen_time() -> f32 { 25.0 }

fn default_bullet_reload_interval() -> f32 { 0.1 }
fn default_bullet_damage() -> f32 { 4.0 }
fn default_bullet_speed() -> f32 { 30.0 }
fn default_bullet_spread() -> f32 { 4f32.to_radians() }
fn default_bullet_lifetime() -> f32 { 6.0 }

pub const MAX_HEALTH_DEFAULT: f32 = 50.0;
