// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use crate::utils::ToStr;

#[derive(Clone, Copy)]
pub enum Security {
	/**
	 * The connection is secure.
	 *
	 * This means that either the TLS certificate is checked and trusted, the
	 * destination IP address is a localhost address, or playing in singleplayer
	 * in which interacting with the network isn't done at all.
	 */
	Secure,

	/**
	 * The connection isn't secure but the destination is on the local network.
	 *
	 * This means that the TLS certificate isn't checked, but the destination IP
	 * address is on the local network. Eavesdropping is still possible but only
	 * on the local network. If you trust the local network, this should be
	 * secure.
	 */
	Local,

	/**
	 * The connection is insecure over the internet.
	 *
	 * This is when the TLS certificate isn't checked and the destination IP
	 * address isn't on the local network.
	 *
	 * This means that all communication can be intercepted and decrypted by
	 * anyone on the path capable of active eavesdropping (being able to read and
	 * modify messages).
	 */
	Insecure,
}

impl ToStr for Security {
	fn to_str(self) -> &'static str {
		match self {
			Security::Secure => "Secure",
			Security::Local => "Local",
			Security::Insecure => "Insecure",
		}
	}
}
