// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::{fs, path::Path};

use glium::{Surface, VertexBuffer, Display, Texture2d, Program, Frame, DrawParameters, index::{NoIndices, PrimitiveType}, uniforms::SamplerWrapFunction, uniform, implement_vertex};
use glutin::surface::WindowSurface;
use glam::Mat4;
use png::ColorType;

use crate::app::filesystem::{Filesystem, FsBase};
use crate::world::checkpoint::{Checkpoint, RADIUS};
use crate::utils::{texture, glium_resize, maths::Circle};

#[derive(Clone, Copy)]
struct Vertex { v_pos: [f32; 2], v_tex_coords: [f32; 2] }
implement_vertex!(Vertex, v_pos, v_tex_coords);

#[derive(Clone, Copy)]
struct Instance { i_pos: [f32; 2], i_t: f32 }
implement_vertex!(Instance, i_pos, i_t);

pub struct CheckpointRenderer {
	v_vbo: VertexBuffer<Vertex>,
	i_vbo: VertexBuffer<Instance>,
	texture: Texture2d,
	program: Program,
	checkpoints: Vec<Instance>,
}

impl CheckpointRenderer {
	pub fn new(display: &Display<WindowSurface>, fs: &Filesystem) -> CheckpointRenderer {
		static VERTICES: [Vertex; 4] = [
			Vertex { v_pos: [-RADIUS,  RADIUS], v_tex_coords: [0.0, 0.0] },
			Vertex { v_pos: [ RADIUS,  RADIUS], v_tex_coords: [1.0, 0.0] },
			Vertex { v_pos: [-RADIUS, -RADIUS], v_tex_coords: [0.0, 1.0] },
			Vertex { v_pos: [ RADIUS, -RADIUS], v_tex_coords: [1.0, 1.0] },
		];

		let vsh = fs::read_to_string(fs.get(FsBase::Static, Path::new("shaders/checkpoint.vsh"))).unwrap();
		let fsh = fs::read_to_string(fs.get(FsBase::Static, Path::new("shaders/checkpoint.fsh"))).unwrap();

		CheckpointRenderer {
			v_vbo: VertexBuffer::immutable(display, &VERTICES).unwrap(),
			i_vbo: VertexBuffer::empty_persistent(display, 0).unwrap(),
			texture: texture::load(display, &fs.get(FsBase::Static, Path::new("textures/checkpoint.png")), ColorType::Rgb),
			program: Program::from_source(display, &vsh, &fsh, None).unwrap(),
			checkpoints: Vec::new(),
		}
	}

	pub fn reset(&mut self, checkpoints: &[Checkpoint], active_checkpoint: Option<usize>, display: &Display<WindowSurface>) {
		self.checkpoints = checkpoints
			.iter()
			.enumerate()
			.map(|(i, checkpoint)| {
				let activated = Some(i) == active_checkpoint;
				Instance {
					i_pos: checkpoint.get_pos().to_array(),
					i_t: activated as u32 as f32,
				}
			})
			.collect();

		glium_resize::vbo_persistent(&mut self.i_vbo, display, self.checkpoints.len());
	}

	pub fn render(&mut self, dt: f32, active_checkpoint: Option<usize>, frame: &mut Frame, params: &DrawParameters, proj_matrix: &Mat4) {
		{
			let mut buf = self.i_vbo.map_write();
			for (i, inst) in self.checkpoints.iter_mut().enumerate() {
				let activated = Some(i) == active_checkpoint;
				inst.i_t = if activated {
					(inst.i_t + dt).min(1.0)
				} else {
					(inst.i_t - dt).max(0.0)
				};
				buf.set(i, *inst);
			}
		}

		let uniforms = uniform! { u_matrix: proj_matrix.to_cols_array_2d(), u_texture: self.texture.sampled().wrap_function(SamplerWrapFunction::Clamp) };
		frame.draw((&self.v_vbo, self.i_vbo.slice(..self.checkpoints.len()).unwrap().per_instance().unwrap()), NoIndices(PrimitiveType::TriangleStrip), &self.program, &uniforms, params).unwrap();
	}
}
