// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use glam::{Vec2, Vec4};

use super::{Particle, temperature::Temperature};

use crate::world::player::direction::Direction;
use crate::utils::maths::decay::Decay;

pub struct Thrust {
	pos: Vec2,
	vel: Vec2,
	size: f32,
	temp: Temperature,
	density: f32,
}

impl Thrust {
	pub fn new(player_pos: Vec2, player_vel: Vec2, dir: Direction) -> Thrust {
		let rotated_dir = Vec2::from(dir).rotate(Vec2::from_angle(rand::random::<f32>() * 0.5 - 0.25)); // Rotates the player's direction by a bit

		Thrust {
			pos: player_pos,
			vel: player_vel - rotated_dir * (rand::random::<f32>() * 6.0 + 8.0),
			size: rand::random::<f32>() * 0.375 + 0.125,
			temp: Temperature::new(rand::random::<f32>() * 1.5 + 2.5),
			density: rand::random::<f32>() * 0.25 + 0.25,
		}
	}
}

impl Particle for Thrust {
	fn get_pos(&self) -> Vec2 { self.pos }
	fn get_size(&self) -> f32 { self.size }
	fn get_colour(&self) -> Vec4 { self.temp.get_colour_rgba() }

	fn update(&mut self, dt: f32) -> bool {
		const DRAG_K: f32 = 0.1;

		let mass = self.size.powi(2) * self.density;
		self.vel.decay(DRAG_K / mass.sqrt(), dt);
		self.pos += self.vel * dt;

		// Particle size decreases linearly over time
		self.size -= 0.125 * dt;
		if self.size <= 0.0 { return false; }

		// The rate of this exponential decay is inversely proportional to its mass (lighter objects cool down faster than heavier ones, or is it size? (might be both))
		self.temp.cool(0.1875 / mass, dt);
		self.temp.visible()
	}
}
