// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::f32::consts::TAU;

use glam::{Vec2, Vec4};

use super::super::Particle;

use crate::world::powerup::RADIUS;
use crate::utils::maths::decay::Decay;

pub struct Reload {
	pos: Vec2,
	size: f32,
	green: f32,
	r: f32,
	max_r: f32,
	theta: f32,
	rot_vel: f32,
	time: f32,
	max_time: f32,
}

impl Reload {
	pub fn new(pos: Vec2) -> Reload {
		let max_time = rand::random::<f32>() + 2.0;
		let r = rand::random::<f32>() * RADIUS;
		Reload {
			pos,
			size: (rand::random::<f32>() + 1.0) * 0.03125,
			green: rand::random::<f32>() * 0.125 + 0.3125,
			r,
			max_r: r + rand::random::<f32>() * 0.25 + 0.625,
			theta: rand::random::<f32>() * TAU,
			rot_vel: rand::random::<f32>() * 2.0 + 1.0,
			time: max_time,
			max_time,
		}
	}
}

impl Particle for Reload {
	fn get_pos(&self) -> Vec2 {
		let (sin, cos) = self.theta.sin_cos();
		self.pos + self.r * Vec2::new(cos, sin)
	}

	fn get_size(&self) -> f32 { self.size }
	fn get_colour(&self) -> Vec4 { Vec4::new(0.5, self.green, 0.125, self.time / self.max_time) }

	fn update(&mut self, dt: f32) -> bool {
		self.r.decay_to(self.max_r, 1.0, dt);
		self.theta -= self.rot_vel * dt;
		self.time -= dt;
		self.time >= 0.0
	}
}
