// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use strum_macros::{EnumCount, EnumIter};

use crate::app::config::action::Action;
use crate::utils::ToStr;

#[derive(Clone, Copy, EnumCount, EnumIter)]
pub(super) enum Category {
	Ui,
	Player,
	Zoom,
	Spectating,
	Debug,
}

impl ToStr for Category {
	fn to_str(self) -> &'static str {
		match self {
			Category::Ui => "UI",
			Category::Player => "Player",
			Category::Zoom => "Camera Zoom",
			Category::Spectating => "Spectator Mode",
			Category::Debug => "Debug",
		}
	}
}

impl ToStr for Action {
	fn to_str(self) -> &'static str {
		match self {
			// UI
			Action::GuiLeave => "GUI Leave",
			Action::PlayingLeave => "Playing Leave",
			Action::ToggleFullscreen => "Fullscreen",
			Action::TeamChatOpen => "Open Team Chat",
			Action::GlobalChatOpen => "Open Global Chat",
			Action::CommandOpen => "Open Command Input",
			Action::ToggleHud => "Toggle HUD",
			Action::StoryTextClose => "Close Story Text",

			// Player
			Action::MoveLeft => "Left",
			Action::MoveRight => "Right",
			Action::MoveDown => "Down",
			Action::MoveUp => "Up",
			Action::Shoot => "Shoot",
			Action::DropFlag => "Drop Flag",

			// Zoom
			Action::ZoomIn => "Zoom In",
			Action::ZoomOut => "Zoom Out",
			Action::ZoomReset => "Reset Zoom",

			// Spectating
			Action::Drag => "Drag",
			Action::FollowPlayer => "Follow Player",
			Action::PrevPlayer => "Previous Player",
			Action::NextPlayer => "Next Player",

			// Debug
			Action::PrintPos => "Print Position",
		}
	}
}

impl From<Action> for Category {
	fn from(action: Action) -> Category {
		match action {
			Action::GuiLeave | Action::PlayingLeave | Action::ToggleFullscreen | Action::TeamChatOpen | Action::GlobalChatOpen | Action::CommandOpen | Action::ToggleHud | Action::StoryTextClose => Category::Ui,
			Action::MoveLeft | Action::MoveRight | Action::MoveDown | Action::MoveUp | Action::Shoot | Action::DropFlag => Category::Player,
			Action::ZoomIn | Action::ZoomOut | Action::ZoomReset => Category::Zoom,
			Action::Drag | Action::FollowPlayer | Action::PrevPlayer | Action::NextPlayer => Category::Spectating,
			Action::PrintPos => Category::Debug,
		}
	}
}
