// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::any::Any;

use glium::Frame;
use egui::{Ui, RichText, Hyperlink};
use winit::event::Event;

use crate::app::{App, config::action::ActionEvent, state::{GameState, Status}};
use crate::app::gui::{GuiState, style::{UiExt, FontSize, H3_SIZE, H3_COLOUR}};
use crate::protocol::info::{Info, InfoType};

pub struct About {
	gui_state: GuiState,
	info: Info,
}

impl About {
	pub fn new() -> About {
		About {
			gui_state: GuiState::new(),
			info: Info::new(),
		}
	}

	fn label_text(s: &str) -> RichText {
		RichText::new(s).font_size(16.0)
	}

	fn label(ui: &mut Ui, s: &str) {
		ui.label(Self::label_text(s).strong());
	}

	fn url(ui: &mut Ui, s: &str) {
		ui.add(Hyperlink::from_label_and_url(Self::label_text(s), s));
	}
}

impl GameState for About {
	fn enable(&mut self, app: &mut App) {
		self.gui_state.enable(&mut app.gui);
	}

	fn push(&mut self, _app: &mut App, msg: Option<Box<dyn Any>>) {
		debug_assert!(msg.is_none());
	}

	fn disable(&mut self, _app: &mut App) {}

	fn action(&mut self, action: ActionEvent) {
		self.gui_state.action(action);
	}

	fn event(&mut self, app: &mut App, event: &Event<()>) {
		self.gui_state.event(app, event);
	}

	fn loop_iter(&mut self, app: &mut App, frame: &mut Frame) -> Status {
		let res = GuiState::update(&mut app.gui, "about", &app.window, |_, ui| {
			ui.h1("About").add();
			ui.horizontal_wrapped(|ui| {
				#[cfg(feature = "flatpak")]
				static FINAL_TEXT: &str = ".";

				// README.md not included in the Flatpak distribution so no point including a link to it
				#[cfg(not(feature = "flatpak"))]
				static FINAL_TEXT: &str = " as well as the attached `README.md`.";

				ui.spacing_mut().item_spacing.x = 0.0; // Removes extra space between link

				Self::label(ui, "Spaceships is a free/libre multiplayer shooter game. In this game you are a spaceship that you can move around with the keyboard and shoot other spaceships to destroy them. The goal of the game depends on the server configuration, which might be getting as many kills as possible, surviving the longest or capturing the most flags.\n\nMost games involve a finite amount of ammo which periodically spawns as ammo crates that can be collected to increase your ammo. Powerups can also spawn, which provide a range of effects including increased speed, reload rate, regeneration rate and a health boost, increased bullet damage, a forcefield that deflects bullets and teleportation so somewhere on the map.\n\nOther gameplay features include portals where you can teleport between areas, and effect zones which provide a powerup effect to all players inside them.\n\nFor more information, see the documentation at ");
				Self::url(ui, "https://spaceships.me/docs/");
				Self::label(ui, FINAL_TEXT);
			});

			ui.h2("Build Information").add();

			self.info.for_each(|key, value, info_type| {
				ui.horizontal_wrapped(|ui| {
					ui.label(RichText::new(key).font_size(H3_SIZE).color(H3_COLOUR));
					let label_text = RichText::new(value).font_size(H3_SIZE);
					match info_type {
						InfoType::Text => ui.label(label_text.strong()),
						InfoType::Url => ui.hyperlink_to(label_text, value),
					}
				});
			});

			if ui.b_back().clicked() {
				self.gui_state.exit();
			}
		});
		self.gui_state.after_update(res);

		app.gui.render(&app.display, frame);

		if self.gui_state.should_exit() { Status::PopState }
		else { Status::Ok }
	}
}
