// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::collections::{BTreeMap, BTreeSet};

use serde::{Serialize, Deserialize};

use super::{World, PlayerId, changes::Changes, event::WorldEvent, DELTA_TIME, player::update::PlayerUpdate, bullet::Bullet, powerup::Effect};

use crate::blocks::Blocks;
use crate::utils::maths::{Circle, CollidingCircle, CollidingRect};

#[derive(Default, Clone, Serialize, Deserialize)]
pub struct WorldUpdate {
	pub pre_events: Vec<WorldEvent>,
	pub players: BTreeMap<PlayerId, PlayerUpdate>,
	pub post_events: Vec<WorldEvent>,
}

impl World {
	pub fn update(&mut self, player_updates: &BTreeMap<PlayerId, PlayerUpdate>, blocks: &Blocks, changes: &mut impl Changes) {
		// Removes all players not in the update
		let old_ids = self.players.keys().copied().collect::<BTreeSet<_>>().difference(&player_updates.keys().copied().collect()).copied().collect::<BTreeSet<_>>();
		self.remove_players(old_ids, changes);

		// Moves the players
		for (&id, &player_update) in player_updates {
			if let Some(player) = self.players.get_mut(&id) {
				player.set_update(player_update);
				player.update(DELTA_TIME, self.config.size, blocks, &self.effects, id);
				player.decrease_cooldown(DELTA_TIME * self.effects.get_mul(id, Effect::Reload));
				if player.shoot() {
					self.bullets.add(Bullet::new(player, id, self.effects.get_mul(id, Effect::Damage), Bullet::random_spread(&mut self.rng)));
					changes.bullet_shot(id, player, &self.effects);
				}
			} else {
				log::warn!("players in update (ids {:?}) don't match players the world (ids {:?})", player_updates.keys().copied().collect::<Vec<_>>(), self.players.keys().copied().collect::<Vec<_>>());
			}
		}

		// Updates the rest of the world
		self.ammo_crates.retain_mut(|ac| {
			if !ac.update(DELTA_TIME) { return false; }
			if let Some((&id, _)) = self.players.iter_mut().find(|(_, player)| player.colliding_rect(ac)) {
				changes.ammo_crate_collected(id, ac.get_pos());
				false
			} else { true }
		});

		changes.clear_powerups();
		self.powerups.retain_mut(|powerup| {
			if !powerup.update(DELTA_TIME) { return false; }
			if let Some((&id, player)) = self.players.iter().find(|(_, player)| player.colliding_circle(powerup)) {
				changes.powerup_collected(id, player, powerup.get_type(), powerup.get_pos());
				false
			} else { true }
		});
		self.effects.update(DELTA_TIME);

		self.bullets.update(DELTA_TIME, &self.get_forcefields());

		for (&id, player) in &mut self.players {
			let collided = self.bullets.collide_player(player, id, self.config.friendly_fire);

			/*
			 * Prevents the same player being "killed twice" (having two Hits
			 * with killed = true when two bullets are colliding and the player
			 * is killed). This is important as scorekeeping and spawning
			 * explosion particles rely on there being one Hit with the player
			 * killed.
			 */
			let mut already_killed = false;
			for bullet in collided {
				player.on_bullet_hit(&bullet);
				let killed = player.is_dead();
				changes.add_hit(bullet, player, id, killed && !already_killed);
				already_killed |= killed;
			}

			if player.is_dead() {
				player.reset();
			}
		}

		changes.update(DELTA_TIME);
		for bullet in self.bullets.collide_blocks(blocks) {
			changes.block_bullet_collision(bullet);
		}

		for flag in &mut self.flags {
			flag.update(DELTA_TIME);
		}
	}
}
