// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::{fs, path::Path};

use glium::{Surface, VertexBuffer, Display, Program, Frame, DrawParameters, index::{NoIndices, PrimitiveType}, uniform, implement_vertex};
use glutin::surface::WindowSurface;
use glam::Mat4;

use super::SpecialArea;

use crate::game::filesystem::{Filesystem, FsBase};
use crate::utils::glium_resize;

#[derive(Clone, Copy)]
struct Vertex { v_pos: [f32; 2] }
implement_vertex!(Vertex, v_pos);

#[derive(Clone, Copy)]
struct Instance { i_pos: [f32; 2], i_size: [f32; 2], i_colour: [f32; 4] }
implement_vertex!(Instance, i_pos, i_size, i_colour);

pub struct Renderer {
	v_vbo: VertexBuffer<Vertex>,
	i_vbo: VertexBuffer<Instance>,
	program: Program,
	count: usize,
}

impl Renderer {
	pub fn new(display: &Display<WindowSurface>, fs: &Filesystem) -> Renderer {
		static VERTICES: [Vertex; 4] = [
			Vertex { v_pos: [0.0, 0.0] },
			Vertex { v_pos: [1.0, 0.0] },
			Vertex { v_pos: [0.0, 1.0] },
			Vertex { v_pos: [1.0, 1.0] },
		];

		let vsh = fs::read_to_string(fs.get(FsBase::Static, Path::new("shaders/special_area.vsh"))).unwrap();
		let fsh = fs::read_to_string(fs.get(FsBase::Static, Path::new("shaders/special_area.fsh"))).unwrap();

		Renderer {
			v_vbo: VertexBuffer::immutable(display, &VERTICES).unwrap(),
			i_vbo: VertexBuffer::empty_persistent(display, 0).unwrap(),
			program: Program::from_source(display, &vsh, &fsh, None).unwrap(),
			count: 0,
		}
	}

	pub fn update(&mut self, areas: &[SpecialArea], display: &Display<WindowSurface>) {
		self.count = areas.len();
		glium_resize::vbo_persistent(&mut self.i_vbo, display, self.count);

		{
			let mut buf = self.i_vbo.map_write();
			for (i, area) in areas.iter().enumerate() {
				buf.set(i, Instance { i_pos: area.pos.to_array(), i_size: area.size.to_array(), i_colour: area.colour.into() });
			}
		}
	}

	pub fn render(&self, frame: &mut Frame, params: &DrawParameters, proj_matrix: &Mat4) {
		let uniforms = uniform! { u_matrix: proj_matrix.to_cols_array_2d() };
		frame.draw((&self.v_vbo, self.i_vbo.slice(..self.count).unwrap().per_instance().unwrap()), NoIndices(PrimitiveType::TriangleStrip), &self.program, &uniforms, params).unwrap();
	}
}
