// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use glam::Vec2;

use crate::world::{World, DELTA_TIME, player::{Player, PlayerId, DPlayer, PlayerController, direction::Direction, forcefield::LocalForcefield}, bullet::Bullet, powerup::{Effect, active_effects::ActiveEffects}};
use crate::blocks::Blocks;
use crate::playing::view::{HealthBar, Sounds, LoopingTrack, LoopingSfx};
use crate::utils::maths::Circle;

pub struct LocalPlayer {
	pub(in crate::world) player: Player,
	pub(in crate::world) last_hurt: f32,
	pub(in crate::world) hbar: HealthBar,
	pub(in crate::world) ffield: Option<LocalForcefield>,
	pub(super) track: LoopingTrack,
	pub(super) block_collision_time: f32,
	pub(super) delta: DPlayer,
}

const BLOCK_COLLISION_SOUND_TIME: f32 = 0.125;

impl LocalPlayer {
	pub(super) fn new(id: PlayerId, player: Player, effects: &ActiveEffects) -> LocalPlayer {
		let hbar = HealthBar::new(player.get_norm_health());
		LocalPlayer {
			player,
			last_hurt: f32::INFINITY,
			hbar,
			ffield: effects.active(id, Effect::Forcefield).then(LocalForcefield::new_existing),
			track: LoopingTrack::nothing(),
			block_collision_time: f32::INFINITY,
			delta: DPlayer::zero(),
		}
	}

	pub fn set_track(&mut self, sounds: &mut Sounds) {
		self.track = sounds.get_looping_track(self.player.get_pos(), self.player.get_vel());
	}

	pub fn inner(&self) -> &Player { &self.player }

	pub fn get_pos(&self, time_diff: f32) -> Vec2 {
		let pos = self.player.get_pos() + self.delta.get_pos();
		let vel = self.get_vel();
		pos + vel * time_diff
	}

	pub fn get_vel(&self) -> Vec2 { self.player.get_vel() + self.delta.get_vel() }
	pub fn get_dir(&self) -> Direction { self.player.get_dir() + self.delta.get_angle() }
	pub fn get_ammo_string(&self) -> String { self.player.get_ammo_count().to_string() }
	pub fn control(&mut self, pcontrol: &PlayerController) { self.player.set_update(pcontrol.get_update()); }

	pub(super) fn update(&mut self, world: &World, blocks: &Blocks, id: PlayerId) -> Option<Bullet> {
		let mut shot = None;

		if self.player.update(DELTA_TIME, world.config.size, blocks, &world.effects, id) {
			self.block_collision_time = 0.0;
		}
		self.player.decrease_cooldown(DELTA_TIME * world.effects.get_mul(id, Effect::Reload));
		if self.player.shoot() {
			shot = Some(Bullet::new(&self.player, id, world.effects.get_mul(id, Effect::Damage), 0.0));
		}
		if let Some(time) = world.effects.get_time(id, Effect::Forcefield) {
			let ffield = self.ffield.get_or_insert_with(LocalForcefield::new);
			ffield.update(time, DELTA_TIME);
		} else {
			self.ffield = None;
		}

		shot
	}

	pub fn update_track(&mut self, dt: f32, time_diff: f32) {
		let sounds = [
			(LoopingSfx::Thrust, self.player.get_input_state().thrusting()),
			(LoopingSfx::BlockPlayer, self.block_collision_time < BLOCK_COLLISION_SOUND_TIME),
		];

		self.track.update(self.get_pos(time_diff), self.get_vel(), &sounds);
		self.block_collision_time += dt;
	}
}
