// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::{fs, path::Path};

use glium::{Surface, VertexBuffer, Display, Program, Frame, DrawParameters, index::{NoIndices, PrimitiveType}, uniforms::SamplerWrapFunction, texture::Texture2dArray, uniform, implement_vertex};
use glutin::surface::WindowSurface;
use glam::Mat4;
use png::ColorType;
use strum::EnumCount;

use super::{Powerup, PowerupType, RADIUS};

use crate::game::filesystem::{Filesystem, FsBase};
use crate::utils::{texture, glium_resize};

#[derive(Clone, Copy)]
struct Vertex { v_pos: [f32; 2], v_tex_coords: [f32; 2] }
implement_vertex!(Vertex, v_pos, v_tex_coords);

#[derive(Clone, Copy)]
struct Instance { i_pos: [f32; 2], i_scale: f32, i_layer: u32 }
implement_vertex!(Instance, i_pos, i_scale, i_layer);

pub struct Renderer {
	v_vbo: VertexBuffer<Vertex>,
	i_vbo: VertexBuffer<Instance>,
	textures: Texture2dArray,
	program: Program,
}

const TEXTURE_RADIUS: f32 = 2.0 * RADIUS;

impl Renderer {
	pub fn new(display: &Display<WindowSurface>, fs: &Filesystem) -> Renderer {
		static VERTICES: [Vertex; 4] = [
			Vertex { v_pos: [-TEXTURE_RADIUS, -TEXTURE_RADIUS], v_tex_coords: [0.0, 0.0] },
			Vertex { v_pos: [ TEXTURE_RADIUS, -TEXTURE_RADIUS], v_tex_coords: [1.0, 0.0] },
			Vertex { v_pos: [-TEXTURE_RADIUS,  TEXTURE_RADIUS], v_tex_coords: [0.0, 1.0] },
			Vertex { v_pos: [ TEXTURE_RADIUS,  TEXTURE_RADIUS], v_tex_coords: [1.0, 1.0] },
		];

		let vsh = fs::read_to_string(fs.get(FsBase::Static, Path::new("shaders/powerup.vsh"))).unwrap();
		let fsh = fs::read_to_string(fs.get(FsBase::Static, Path::new("shaders/powerup.fsh"))).unwrap();

		Renderer {
			v_vbo: VertexBuffer::immutable(display, &VERTICES).unwrap(),
			i_vbo: VertexBuffer::empty_persistent(display, 0).unwrap(),
			textures: texture::load_array(display, &fs.get(FsBase::Static, Path::new("textures/powerups.png")), ColorType::Rgb, PowerupType::COUNT as u32),
			program: Program::from_source(display, &vsh, &fsh, None).unwrap(),
		}
	}

	pub fn render(&mut self, powerups: &[Powerup], display: &Display<WindowSurface>, frame: &mut Frame, params: &DrawParameters, proj_matrix: &Mat4, time_shift: f32) {
		let count = powerups.len();
		glium_resize::vbo_persistent(&mut self.i_vbo, display, count);

		{
			let mut buf = self.i_vbo.map_write();
			for (i, powerup) in powerups.iter().enumerate() {
				buf.set(i, Instance { i_pos: powerup.pos.to_array(), i_scale: Powerup::get_scale(powerup.time + time_shift), i_layer: powerup.typ as u32 });
			}
		}

		let uniforms = uniform! { u_matrix: proj_matrix.to_cols_array_2d(), u_texture: self.textures.sampled().wrap_function(SamplerWrapFunction::Clamp) };
		frame.draw((&self.v_vbo, self.i_vbo.slice(..count).unwrap().per_instance().unwrap()), NoIndices(PrimitiveType::TriangleStrip), &self.program, &uniforms, params).unwrap();
	}
}
