// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::collections::BTreeMap;

use serde::{Serialize, Deserialize};
use strum::EnumCount;
use glam::Vec2;

use super::Effect;

use crate::world::{Changes, player::PlayerId};
#[cfg(feature = "client")] use crate::world::MAX_PLAYERS;

#[derive(Debug, Default, Clone, Copy, Serialize, Deserialize)]
pub struct EffectInfo {
	pub time: f32,
	pub power: f32,
}

impl EffectInfo {
	pub fn active(self) -> bool {
		self.time > 0.0
	}

	pub fn get_power(self) -> Option<f32> {
		self.active().then_some(self.power)
	}

	#[cfg(feature = "client")]
	pub fn normalise(&mut self, effect: Effect) {
		if !self.active() {
			self.power = effect.default_power();
		}
	}
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct ActiveEffects(BTreeMap<PlayerId, [EffectInfo; Effect::COUNT]>);

impl ActiveEffects {
	pub fn new() -> ActiveEffects {
		ActiveEffects(BTreeMap::new())
	}

	#[cfg(feature = "client")]
	pub fn validate(&self) -> Result<(), &'static str> {
		if self.0.len() <= MAX_PLAYERS { Ok(()) }
		else { Err("too many players") }
	}

	pub fn remove_player(&mut self, id: PlayerId) {
		self.0.remove(&id);
	}

	fn get_info(&self, id: PlayerId, effect: Effect) -> Option<EffectInfo> {
		Some(self.0.get(&id)?[effect as usize])
	}

	#[cfg(feature = "client")]
	pub fn active(&self, id: PlayerId, effect: Effect) -> bool {
		self.get_info(id, effect).is_some_and(EffectInfo::active)
	}

	#[cfg(feature = "client")]
	pub fn get_time(&self, id: PlayerId, effect: Effect) -> Option<f32> {
		let info = self.get_info(id, effect)?;
		info.active().then_some(info.time)
	}

	/**
	 * Returns the "power" of the effect if the player has that effect, otherwise
	 * None.
	 *
	 * The interpretation of this "power" depends on the type of effect. For
	 * forcefields, it's the strength of the repulsion and for everything else
	 * it's a multiplier.
	 */
	pub fn get_power(&self, id: PlayerId, effect: Effect) -> Option<f32> {
		self.get_info(id, effect)?.get_power()
	}

	/**
	 * Convenience method for effects when the power is a multiplier, to which a
	 * default of 1 is returned if the effect isn't active.
	 */
	pub fn get_mul(&self, id: PlayerId, effect: Effect) -> f32 {
		self.get_power(id, effect).unwrap_or(1.0)
	}

	pub fn add_effect(&mut self, id: PlayerId, effect: Effect, info: EffectInfo, changes: &mut impl Changes, player_pos: Vec2, play_sound: bool) {
		let effects = self.0.entry(id).or_default();
		let old_info = &mut effects[effect as usize];
		changes.effect_event(id, effect, *old_info, info, player_pos, play_sound);
		*old_info = info;
	}

	/**
	 * Returns an iterator of all player ids and the corresponding time of all
	 * players with `effect`.
	 */
	pub fn player_iter_with_effect(&self, effect: Effect) -> impl Iterator<Item = (PlayerId, EffectInfo)> + '_ {
		self.0.iter().filter_map(move |(&id, effects)| {
			let info = effects[effect as usize];
			info.active().then_some((id, info))
		})
	}

	pub fn update(&mut self, dt: f32) {
		self.0.retain(|_, effects| {
			let mut has_active = false;
			for effect in effects {
				effect.time -= dt;
				has_active |= effect.active();
			}
			has_active
		});
	}
}
