// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
#[cfg(feature = "client")] mod renderer;
pub mod active_effects;

#[cfg(feature = "client")] pub use renderer::Renderer as PowerupRenderer;

use strum_macros::{EnumCount as EnumCountMacro, EnumIter};
use serde::{Serialize, Deserialize};
use glam::Vec2;

use crate::utils::maths::Circle;

#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize, EnumCountMacro, EnumIter)]
pub enum Effect {
	Speed,
	Reload,
	Regeneration,
	Damage,
	Forcefield,
}

#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize, EnumCountMacro, EnumIter)]
pub enum PowerupType {
	Speed,
	Reload,
	Health,
	Damage,
	Forcefield,
	Teleportation,
}

pub enum PowerType {
	Multiplicative,
	Additive,
}

impl Effect {
	pub fn try_from_type(typ: PowerupType) -> Option<Effect> {
		match typ {
			PowerupType::Speed => Some(Effect::Speed),
			PowerupType::Reload => Some(Effect::Reload),
			PowerupType::Health => Some(Effect::Regeneration),
			PowerupType::Damage => Some(Effect::Damage),
			PowerupType::Forcefield => Some(Effect::Forcefield),
			PowerupType::Teleportation => None,
		}
	}

	pub fn power_type(self) -> PowerType {
		match self {
			Effect::Speed | Effect::Reload | Effect::Regeneration | Effect::Damage => PowerType::Multiplicative,
			Effect::Forcefield => PowerType::Additive,
		}
	}

	#[cfg(feature = "client")]
	pub fn default_power(self) -> f32 {
		match self.power_type() {
			PowerType::Multiplicative => 1.0,
			PowerType::Additive => 0.0,
		}
	}
}

impl From<Effect> for PowerupType {
	fn from(effect: Effect) -> PowerupType {
		match effect {
			Effect::Speed => PowerupType::Speed,
			Effect::Reload => PowerupType::Reload,
			Effect::Regeneration => PowerupType::Health,
			Effect::Damage => PowerupType::Damage,
			Effect::Forcefield => PowerupType::Forcefield,
		}
	}
}

pub const RADIUS: f32 = 1.0 / 3.0;
const MAX_TIME: f32 = 10.0 * 60.0;
const FADE_IN_TIME: f32 = 1.0;
const FADE_OUT_TIME: f32 = 30.0;

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Powerup {
	pos: Vec2,
	scale: f32,
	time: f32,
	typ: PowerupType,
}

impl Powerup {
	pub fn new(pos: Vec2, typ: PowerupType) -> Powerup {
		Powerup { pos, scale: 0.0, time: 0.0, typ }
	}

	pub fn get_type(&self) -> PowerupType { self.typ }

	pub fn update(&mut self, dt: f32) -> bool {
		self.time += dt;
		self.scale = Powerup::get_scale(self.time);
		self.time <= MAX_TIME
	}

	fn get_scale(time: f32) -> f32 {
		if time < FADE_IN_TIME {
			let f = time / FADE_IN_TIME;
			f * (2.0 - f)
		} else if time > MAX_TIME - FADE_OUT_TIME {
			let f = (MAX_TIME - time) / FADE_OUT_TIME;
			f * (2.0 - f)
		} else {
			1.0
		}
	}
}

impl Circle for Powerup {
	fn get_pos(&self) -> Vec2 { self.pos }
	fn get_radius(&self) -> f32 { RADIUS * self.scale }
}
