// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use glam::Vec2;

use super::{Direction, BITS, COUNT};

const QUAD_COUNT: u16 = COUNT / 4;
const QUAD_EXTRACT_MASK: u16 = (1 << (BITS - 1)) | (1 << (BITS - 2));

// TABLE[i] = sin(i / 1024 * π/2)
static TABLE: [f32; QUAD_COUNT as usize + 1] = [
	0.0, 0.0015339802, 0.0030679568, 0.004601926, 0.0061358847, 0.0076698293, 0.009203754, 0.010737659, 0.012271538, 0.0138053885, 0.015339207, 0.016872989, 0.01840673, 0.019940428, 0.021474082, 0.023007682, 0.024541229, 0.026074719, 0.027608145, 0.02914151, 0.030674804, 0.03220803, 0.033741176, 0.035274237, 0.036807224, 0.03834012, 0.03987293, 0.041405644, 0.04293826, 0.044470772, 0.04600318, 0.047535487, 0.049067676, 0.05059975, 0.05213171, 0.05366354, 0.055195246, 0.05672682, 0.058258265, 0.059789572, 0.06132074, 0.06285176, 0.064382635, 0.06591336, 0.06744392, 0.06897433, 0.070504576, 0.07203466, 0.07356457, 0.075094305, 0.076623864, 0.078153245, 0.07968244, 0.08121145, 0.08274027, 0.08426889, 0.08579732, 0.087325536, 0.08885355, 0.09038136, 0.091908954, 0.09343634, 0.0949635, 0.096490435, 0.09801714, 0.099543616, 0.10106987, 0.10259587, 0.10412164, 0.105647154, 0.10717243, 0.108697444, 0.110222206, 0.11174671, 0.11327095, 0.11479493, 0.116318636, 0.11784206, 0.119365215, 0.12088809, 0.12241068, 0.12393298, 0.12545498, 0.1269767, 0.12849812, 0.13001922, 0.13154003, 0.13306053, 0.13458072, 0.13610058, 0.13762012, 0.13913935, 0.14065824, 0.1421768, 0.14369504, 0.14521293, 0.14673047, 0.14824769, 0.14976454, 0.15128104, 0.15279719, 0.15431298, 0.15582842, 0.15734346, 0.15885815, 0.16037247, 0.1618864, 0.16339995, 0.16491313, 0.1664259, 0.16793829, 0.1694503, 0.1709619, 0.17247309, 0.17398387, 0.17549427, 0.17700422, 0.17851378, 0.18002291, 0.18153162, 0.18303989, 0.18454774, 0.18605515, 0.18756212, 0.18906866, 0.19057477, 0.19208041, 0.19358559, 0.19509032, 0.19659461, 0.1980984, 0.19960175, 0.20110464, 0.20260705, 0.20410897, 0.20561042, 0.20711139, 0.20861185, 0.21011184, 0.21161133, 0.21311033, 0.21460882, 0.2161068, 0.21760428, 0.21910124, 0.2205977, 0.22209363, 0.22358905, 0.22508392, 0.22657827, 0.22807209, 0.22956537, 0.23105812, 0.23255032, 0.23404196, 0.23553306, 0.2370236, 0.2385136, 0.24000302, 0.24149188, 0.2429802, 0.2444679, 0.24595505, 0.24744162, 0.24892761, 0.250413, 0.2518978, 0.25338206, 0.25486568, 0.2563487, 0.2578311, 0.25931293, 0.26079413, 0.2622747, 0.2637547, 0.26523405, 0.26671278, 0.26819086, 0.2696683, 0.27114516, 0.27262136, 0.27409694, 0.27557182, 0.2770461, 0.2785197, 0.27999264, 0.28146493, 0.28293657, 0.28440756, 0.28587785, 0.28734747, 0.2888164, 0.29028466, 0.29175228, 0.29321918, 0.2946854, 0.2961509, 0.29761574, 0.29907984, 0.30054325, 0.30200595, 0.30346796, 0.30492926, 0.3063898, 0.30784968, 0.30930877, 0.31076714, 0.3122248, 0.31368175, 0.31513795, 0.3165934, 0.3180481, 0.31950203, 0.32095525, 0.3224077, 0.32385936, 0.32531032, 0.32676047, 0.32820985, 0.32965845, 0.3311063, 0.3325534, 0.33399966, 0.33544517, 0.33688986, 0.33833376, 0.33977687, 0.34121922, 0.34266073, 0.34410143, 0.34554133, 0.34698042, 0.34841868, 0.34985614, 0.35129276, 0.35272858, 0.35416353, 0.35559767, 0.357031, 0.3584634, 0.35989505, 0.3613258, 0.36275575, 0.3641848, 0.365613, 0.36704037, 0.36846682, 0.36989245, 0.3713172, 0.37274107, 0.37416407, 0.37558618, 0.37700742, 0.37842774, 0.3798472, 0.3812658, 0.38268346, 0.3841002, 0.38551608, 0.38693103, 0.38834503, 0.38975817, 0.39117038, 0.39258167, 0.39399207, 0.3954015, 0.39681, 0.39821756, 0.3996242, 0.4010299, 0.40243468, 0.4038385, 0.40524134, 0.40664324, 0.40804416, 0.40944415, 0.4108432, 0.41224125, 0.41363832, 0.41503444, 0.41642958, 0.4178237, 0.4192169, 0.4206091, 0.4220003, 0.42339048, 0.4247797, 0.4261679, 0.42755508, 0.4289413, 0.4303265, 0.43171066, 0.43309385, 0.434476, 0.43585712, 0.43723717, 0.43861625, 0.43999428, 0.4413713, 0.44274724, 0.44412217, 0.44549602, 0.44686884, 0.4482406, 0.44961134, 0.450981, 0.4523496, 0.45371714, 0.45508358, 0.45644897, 0.45781332, 0.45917657, 0.46053872, 0.46189982, 0.46325982, 0.46461868, 0.4659765, 0.46733323, 0.46868885, 0.47004336, 0.47139674, 0.47274905, 0.4741002, 0.47545028, 0.47679925, 0.47814706, 0.47949374, 0.4808393, 0.48218375, 0.4835271, 0.48486924, 0.4862103, 0.48755017, 0.48888892, 0.4902265, 0.49156293, 0.49289823, 0.49423233, 0.4955653, 0.4968971, 0.49822766, 0.4995571, 0.50088537, 0.50221246, 0.50353837, 0.50486314, 0.50618666, 0.507509, 0.5088302, 0.51015013, 0.5114689, 0.51278645, 0.51410276, 0.5154179, 0.5167318, 0.5180445, 0.519356, 0.52066624, 0.5219753, 0.5232831, 0.5245897, 0.52589506, 0.52719915, 0.52850205, 0.52980363, 0.531104, 0.5324032, 0.533701, 0.53499764, 0.53629297, 0.5375871, 0.53887993, 0.5401715, 0.54146177, 0.5427508, 0.54403853, 0.545325, 0.5466102, 0.54789406, 0.5491767, 0.550458, 0.55173796, 0.5530167, 0.5542941, 0.55557024, 0.55684507, 0.5581185, 0.5593907, 0.5606616, 0.56193113, 0.56319934, 0.5644663, 0.5657318, 0.5669961, 0.568259, 0.56952053, 0.57078075, 0.5720396, 0.57329714, 0.5745534, 0.5758082, 0.5770617, 0.5783138, 0.5795646, 0.580814, 0.582062, 0.5833087, 0.58455396, 0.5857979, 0.58704036, 0.5882816, 0.5895213, 0.5907597, 0.5919967, 0.59323233, 0.5944665, 0.5956993, 0.59693074, 0.59816074, 0.5993893, 0.6006165, 0.6018423, 0.6030666, 0.60428953, 0.605511, 0.6067311, 0.6079498, 0.60916704, 0.6103828, 0.6115972, 0.6128101, 0.6140216, 0.61523163, 0.6164402, 0.61764735, 0.61885303, 0.6200572, 0.62126, 0.62246126, 0.6236611, 0.6248595, 0.6260564, 0.6272518, 0.6284458, 0.62963825, 0.6308293, 0.63201874, 0.6332068, 0.63439333, 0.63557833, 0.63676184, 0.6379439, 0.63912445, 0.6403035, 0.64148104, 0.64265704, 0.64383155, 0.6450046, 0.64617604, 0.647346, 0.64851445, 0.6496813, 0.6508467, 0.65201056, 0.65317285, 0.6543336, 0.65549284, 0.65665054, 0.6578067, 0.6589613, 0.66011435, 0.66126585, 0.6624158, 0.6635642, 0.664711, 0.66585624, 0.66699994, 0.6681421, 0.66928256, 0.67042154, 0.671559, 0.6726948, 0.673829, 0.6749617, 0.67609274, 0.6772222, 0.6783501, 0.6794763, 0.680601, 0.68172413, 0.6828456, 0.68396544, 0.6850837, 0.6862003, 0.68731534, 0.68842876, 0.68954057, 0.6906507, 0.6917593, 0.6928662, 0.69397146, 0.69507515, 0.6961772, 0.69727755, 0.6983763, 0.6994734, 0.7005688, 0.7016626, 0.70275474, 0.70384526, 0.70493406, 0.70602125, 0.70710677, 0.7081907, 0.70927286, 0.7103534, 0.7114322, 0.7125094, 0.7135849, 0.7146587, 0.71573085, 0.7168013, 0.71787006, 0.71893716, 0.72000253, 0.72106624, 0.7221282, 0.7231885, 0.7242471, 0.725304, 0.7263592, 0.72741264, 0.7284644, 0.7295144, 0.73056275, 0.7316094, 0.7326543, 0.7336975, 0.7347389, 0.7357786, 0.7368166, 0.7378528, 0.73888737, 0.73992014, 0.7409512, 0.74198043, 0.74300796, 0.74403375, 0.74505776, 0.7460801, 0.7471006, 0.7481194, 0.7491364, 0.75015163, 0.75116515, 0.7521769, 0.7531868, 0.754195, 0.7552014, 0.75620604, 0.7572089, 0.7582099, 0.7592092, 0.7602067, 0.7612024, 0.7621963, 0.7631884, 0.76417875, 0.7651673, 0.766154, 0.76713896, 0.7681221, 0.76910335, 0.7700829, 0.7710606, 0.7720364, 0.77301043, 0.7739827, 0.7749531, 0.7759217, 0.7768885, 0.7778534, 0.7788165, 0.7797778, 0.7807373, 0.7816948, 0.78265065, 0.78360456, 0.7845566, 0.78550684, 0.7864552, 0.78740174, 0.7883464, 0.78928924, 0.7902302, 0.79116935, 0.79210657, 0.793042, 0.7939755, 0.79490715, 0.7958369, 0.79676485, 0.79769087, 0.798615, 0.7995373, 0.80045766, 0.80137616, 0.8022928, 0.8032075, 0.8041204, 0.80503136, 0.8059404, 0.8068476, 0.80775285, 0.8086562, 0.8095577, 0.81045717, 0.8113549, 0.8122506, 0.81314445, 0.8140363, 0.8149263, 0.81581444, 0.8167006, 0.8175848, 0.81846714, 0.81934756, 0.820226, 0.82110256, 0.82197714, 0.8228498, 0.8237205, 0.8245893, 0.82545614, 0.82632107, 0.827184, 0.82804507, 0.82890415, 0.82976127, 0.8306164, 0.83146966, 0.83232087, 0.8331702, 0.8340175, 0.8348629, 0.8357063, 0.83654773, 0.8373872, 0.8382247, 0.83906025, 0.8398938, 0.84072536, 0.84155494, 0.8423826, 0.84320825, 0.8440319, 0.8448536, 0.84567326, 0.8464909, 0.84730667, 0.84812033, 0.8489321, 0.84974176, 0.8505495, 0.8513552, 0.8521589, 0.85296065, 0.8537603, 0.854558, 0.8553537, 0.85614735, 0.856939, 0.85772866, 0.8585163, 0.85930187, 0.8600854, 0.86086696, 0.86164653, 0.862424, 0.8631994, 0.86397284, 0.86474425, 0.8655136, 0.866281, 0.86704624, 0.8678095, 0.8685707, 0.86932987, 0.87008697, 0.8708421, 0.8715951, 0.87234604, 0.873095, 0.8738419, 0.87458664, 0.87532943, 0.87607014, 0.87680876, 0.8775453, 0.8782798, 0.8790123, 0.8797426, 0.88047093, 0.88119715, 0.8819213, 0.8826434, 0.8833633, 0.88408124, 0.8847971, 0.88551086, 0.88622254, 0.88693213, 0.88763964, 0.88834506, 0.88904834, 0.8897496, 0.89044875, 0.89114577, 0.8918407, 0.89253354, 0.8932243, 0.893913, 0.8945995, 0.89528394, 0.8959663, 0.8966465, 0.8973246, 0.8980006, 0.8986745, 0.8993463, 0.90001595, 0.90068346, 0.9013489, 0.90201217, 0.9026733, 0.90333235, 0.9039893, 0.9046441, 0.90529674, 0.90594727, 0.9065957, 0.907242, 0.90788615, 0.90852815, 0.909168, 0.9098057, 0.9104413, 0.91107476, 0.91170603, 0.9123352, 0.9129622, 0.9135871, 0.9142098, 0.9148303, 0.9154487, 0.916065, 0.9166791, 0.91729105, 0.9179008, 0.9185084, 0.9191139, 0.9197172, 0.92031825, 0.9209172, 0.92151403, 0.92210865, 0.9227011, 0.92329144, 0.9238795, 0.9244655, 0.92504925, 0.9256308, 0.9262102, 0.9267875, 0.92736256, 0.9279354, 0.9285061, 0.9290746, 0.9296409, 0.93020505, 0.930767, 0.93132675, 0.9318843, 0.9324396, 0.9329928, 0.9335438, 0.9340926, 0.93463916, 0.9351835, 0.9357257, 0.93626565, 0.93680346, 0.937339, 0.93787235, 0.93840355, 0.9389325, 0.9394592, 0.9399838, 0.9405061, 0.9410262, 0.94154406, 0.94205976, 0.9425732, 0.9430845, 0.94359344, 0.94410026, 0.9446049, 0.9451072, 0.94560736, 0.94610524, 0.9466009, 0.9470944, 0.94758564, 0.94807464, 0.9485614, 0.9490459, 0.9495282, 0.9500082, 0.95048606, 0.95096165, 0.951435, 0.95190614, 0.952375, 0.95284164, 0.953306, 0.9537682, 0.9542281, 0.95468575, 0.9551412, 0.95559436, 0.95604527, 0.9564939, 0.95694035, 0.9573845, 0.95782644, 0.9582661, 0.9587035, 0.95913863, 0.95957154, 0.9600022, 0.96043056, 0.9608567, 0.9612805, 0.9617021, 0.9621214, 0.9625385, 0.96295327, 0.9633658, 0.96377605, 0.96418405, 0.9645898, 0.96499324, 0.96539444, 0.9657934, 0.96619, 0.9665844, 0.96697646, 0.9673663, 0.9677538, 0.9681391, 0.96852213, 0.9689028, 0.96928126, 0.9696574, 0.97003126, 0.97040284, 0.97077215, 0.9711392, 0.9715039, 0.97186637, 0.9722265, 0.97258437, 0.97293997, 0.97329324, 0.97364426, 0.97399294, 0.97433937, 0.9746835, 0.97502536, 0.97536486, 0.9757021, 0.9760371, 0.97636974, 0.97670007, 0.97702813, 0.97735393, 0.97767735, 0.9779985, 0.9783174, 0.97863394, 0.9789482, 0.97926015, 0.9795698, 0.9798771, 0.9801822, 0.9804849, 0.9807853, 0.9810834, 0.9813792, 0.9816727, 0.9819639, 0.9822527, 0.9825393, 0.98282355, 0.9831055, 0.9833851, 0.9836624, 0.98393744, 0.9842101, 0.98448044, 0.9847485, 0.98501426, 0.98527765, 0.9855387, 0.9857975, 0.98605394, 0.9863081, 0.9865599, 0.98680943, 0.98705655, 0.9873014, 0.98754394, 0.98778415, 0.988022, 0.9882576, 0.9884908, 0.9887217, 0.98895025, 0.9891765, 0.98940045, 0.989622, 0.9898413, 0.9900582, 0.9902728, 0.9904851, 0.990695, 0.99090266, 0.99110794, 0.99131083, 0.99151146, 0.99170977, 0.9919057, 0.99209934, 0.9922906, 0.99247956, 0.9926661, 0.9928504, 0.99303234, 0.993212, 0.99338925, 0.9935641, 0.99373674, 0.993907, 0.9940749, 0.99424046, 0.99440366, 0.9945646, 0.99472314, 0.9948793, 0.9950332, 0.9951847, 0.9953339, 0.9954808, 0.99562526, 0.9957674, 0.99590725, 0.9960447, 0.9961798, 0.9963126, 0.99644303, 0.9965711, 0.9966969, 0.99682033, 0.9969414, 0.99706006, 0.99717647, 0.99729043, 0.99740213, 0.99751145, 0.99761844, 0.9977231, 0.9978254, 0.9979253, 0.99802285, 0.9981181, 0.998211, 0.99830157, 0.9983897, 0.99847555, 0.99855906, 0.99864024, 0.99871904, 0.99879545, 0.99886954, 0.9989413, 0.9990107, 0.99907774, 0.9991424, 0.99920475, 0.9992648, 0.9993224, 0.99937767, 0.9994306, 0.9994812, 0.9995294, 0.9995753, 0.9996188, 0.99966, 0.9996988, 0.9997353, 0.9997694, 0.99980116, 0.9998306, 0.99985766, 0.99988234, 0.9999047, 0.9999247, 0.99994236, 0.9999576, 0.9999706, 0.99998116, 0.9999894, 0.9999953, 0.9999988, 1.0,
];

#[inline]
pub(super) fn sin_1q(i: u16) -> f32 {
	TABLE[i as usize]
}

#[inline]
pub(super) fn sin_2q(i: u16) -> f32 {
	TABLE[(QUAD_COUNT - i) as usize]
}

#[inline]
pub(super) fn sin_3q(i: u16) -> f32 {
	-TABLE[i as usize]
}

#[inline]
pub(super) fn sin_4q(i: u16) -> f32 {
	-TABLE[(QUAD_COUNT - i) as usize]
}

// cos(θ) = sin(θ + π/2)
#[inline] pub(super) fn cos_1q(i: u16) -> f32 { sin_2q(i) }
#[inline] pub(super) fn cos_2q(i: u16) -> f32 { sin_3q(i) }
#[inline] pub(super) fn cos_3q(i: u16) -> f32 { sin_4q(i) }
#[inline] pub(super) fn cos_4q(i: u16) -> f32 { sin_1q(i) }

impl From<Direction> for Vec2 {
	fn from(dir: Direction) -> Vec2 {
		const Q1: u16 = 0;
		const Q2: u16 = QUAD_COUNT;
		const Q3: u16 = QUAD_COUNT * 2;

		let quad = dir.0 & QUAD_EXTRACT_MASK;
		let i = dir.0 - quad;
		match quad {
			Q1 => Vec2::new(cos_1q(i), sin_1q(i)),
			Q2 => Vec2::new(cos_2q(i), sin_2q(i)),
			Q3 => Vec2::new(cos_3q(i), sin_3q(i)),
			_  => Vec2::new(cos_4q(i), sin_4q(i)),
		}
	}
}
