// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::ops::{Add, Sub, AddAssign};

use glam::Vec2;

use super::{Player, direction::Angle, super::DELTA_TIME};

use crate::utils::maths::decay::Decay;

#[derive(Default)]
pub struct DPlayer {
	pos: Vec2,
	vel: Vec2,
	angle: Angle,
}

impl DPlayer {
	pub fn zero() -> DPlayer { DPlayer::default() }

	pub fn get_pos(&self) -> Vec2 { self.pos }
	pub fn get_vel(&self) -> Vec2 { self.vel }
	pub fn get_angle(&self) -> Angle { self.angle }

	pub fn decay(&mut self, rate: f32) {
		/*
		 * Decays the direction faster as a jump in the position can feel very
		 * weird, but a jump in the direction can be normal gameplay.
		 */
		self.pos.decay(rate, DELTA_TIME);
		self.vel.decay(rate, DELTA_TIME);
		self.angle.decay(rate * 8.0, DELTA_TIME);
	}

	pub fn reset_angle(&mut self) {
		self.angle = Angle::default();
	}
}

impl Sub for &Player {
	type Output = DPlayer;

	fn sub(self, other: &Player) -> DPlayer {
		DPlayer {
			pos: self.pos - other.pos,
			vel: self.vel - other.vel,
			angle: self.dir - other.dir,
		}
	}
}

impl Add<&DPlayer> for &Player {
	type Output = Player;

	fn add(self, delta: &DPlayer) -> Player {
		Player {
			pos: self.pos + delta.pos,
			vel: self.vel + delta.vel,
			dir: self.dir + delta.angle,
			.. self.clone()
		}
	}
}

impl AddAssign<&DPlayer> for DPlayer {
	fn add_assign(&mut self, rhs: &DPlayer) {
		self.pos += rhs.pos;
		self.vel += rhs.vel;
		self.angle += rhs.angle;
	}
}
