// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
mod id;
pub mod update;
#[cfg(feature = "client")] mod renderer;
#[cfg(feature = "client")] mod controller;
mod input_state;
pub mod ammo_count;
pub mod forcefield;
mod collision;
#[cfg(feature = "client")] mod delta;
pub mod direction;
mod style;

pub use id::PlayerId;
#[cfg(feature = "client")] pub use controller::Controller as PlayerController;
#[cfg(feature = "client")] pub use renderer::Renderer as PlayerRenderer;
#[cfg(feature = "client")] pub use delta::DPlayer;
pub use style::{PlayerStyle, PlayerName};
#[cfg(feature = "client")] pub use style::{PlayerNameRef, PlayerNameMut};

use glam::Vec2;
use serde::{Serialize, Deserialize};

use input_state::InputState;
use ammo_count::AmmoCount;
use update::PlayerUpdate;
use collision::RESTITUTION_K;
use direction::Direction;

use super::{bullet::Bullet, powerup::{Effect, active_effects::ActiveEffects}, team::TeamId};
#[cfg(feature = "client")] use super::colour::Colour;

use crate::utils::maths::Circle;
use crate::blocks::Blocks;

// The hitbox of the spaceship is a circle to make things easy
// Annoyingly, square root isn't a constant function so I need to hard-code this value in
pub const RADIUS: f32 = 0.44603103; // (0.625 / PI).sqrt(), using 0.625 from results/area.py which calculates the area of the model
const BULLET_INTERVAL: f32 = 0.1;
pub const MAX_HEALTH: f32 = 50.0;
#[cfg(feature = "client")] pub const MIN_DIST_TO_EDGE: f32 = 0.25;

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Player {
	pos: Vec2, vel: Vec2,
	dir: Direction,
	health: f32,
	cooldown: f32,
	ammo: AmmoCount,
	input_state: InputState,
	style: PlayerStyle,
	team: Option<TeamId>,
}

impl Player {
	pub fn new(style: PlayerStyle, team: Option<TeamId>) -> Player {
		Player {
			pos: Vec2::ZERO, vel: Vec2::ZERO, dir: Direction::default(),
			health: MAX_HEALTH, cooldown: 0.0, ammo: AmmoCount::default(), input_state: InputState::default(),
			style, team,
		}
	}

	pub fn get_vel(&self) -> Vec2 { self.vel }
	#[cfg(feature = "client")] pub fn get_norm_health(&self) -> f32 { (self.health / MAX_HEALTH).clamp(0.0, 2.0) }
	pub fn get_dir(&self) -> Direction { self.dir }
	pub fn get_input_state(&self) -> InputState { self.input_state }
	pub fn get_ammo_count(&self) -> AmmoCount { self.ammo }
	#[cfg(feature = "client")] pub fn get_style(&self) -> PlayerStyle { self.style.clone() }
	pub fn get_style_mut(&mut self) -> &mut PlayerStyle { &mut self.style }
	#[cfg(feature = "client")] pub fn get_colour(&self) -> Colour { self.style.colour }
	pub fn get_team(&self) -> Option<TeamId> { self.team }

	pub fn set_pos(&mut self, pos: Vec2) { self.pos = pos; }
	pub fn set_vel(&mut self, vel: Vec2) { self.vel = vel; }
	pub fn set_health(&mut self, health: f32) { self.health = health; }
	pub fn set_ammo_count(&mut self, ammo: AmmoCount) { self.ammo = ammo; }
	pub fn set_team(&mut self, team: Option<TeamId>) { self.team= team; }

	pub fn set_update(&mut self, update: PlayerUpdate) {
		self.dir = update.dir;
		self.input_state = update.input_state;
	}

	pub fn add_health(&self, amount: f32) -> f32 {
		(self.health + amount).min(MAX_HEALTH * 2.0)
	}

	pub fn on_bullet_hit(&mut self, bullet: &Bullet) {
		self.health -= bullet.get_damage();
	}

	pub fn is_dead(&self) -> bool { self.health <= 0.0 }

	pub fn reset(&mut self) {
		self.health = MAX_HEALTH;
		self.cooldown = 0.0;
	}

	pub fn update(&mut self, dt: f32, world_size: Vec2, blocks: &Blocks, effects: &ActiveEffects, id: PlayerId) -> bool {
		let speed_mul = effects.get_mul(id, Effect::Speed);
		let prev_pos = self.pos;

		let acc = self.get_acc(dt, speed_mul);
		self.vel += acc * dt;
		self.pos += self.vel * dt;

		self.health = (self.health + dt * effects.get_mul(id, Effect::Regeneration)).min(self.health.max(MAX_HEALTH));

		/*
		 * Prevents the player from leaving the world. If the player does leave,
		 * then put them back and negate one component of their velocity.
		 */
		for i in 0..2 {
			if self.pos[i] < -world_size[i] / 2.0 + RADIUS { self.pos[i] = -world_size[i] / 2.0 + RADIUS; self.vel[i] *= -RESTITUTION_K; }
			else if self.pos[i] > world_size[i] / 2.0 - RADIUS { self.pos[i] = world_size[i] / 2.0 - RADIUS; self.vel[i] *= -RESTITUTION_K; }
		}

		self.collide_blocks(world_size, blocks, prev_pos, dt)
	}

	fn get_acc(&self, dt: f32, speed_mul: f32) -> Vec2 {
		const MASS: f32 = 1.0;
		const MAX_SPEED: f32 = 12.0;
		const DRAG_A: f32 = 0.5;
		const DRAG_B: f32 = 1.0;

		let move_dir = self.input_state.get_move_dir();

		let max_speed = MAX_SPEED * speed_mul;
		let thrust_size = DRAG_A * max_speed * (max_speed + DRAG_B); // The thrust size required to exactly oppose the drag at the max speed
		let mut force = move_dir * thrust_size;

		/*
		 * The vel * speed factor is equivalent to dir * speed² (where dir is the
		 * direction of the velocity), which is based on the air resistance
		 * equation. The vel factor without the speed has the force proportional
		 * to the speed directly, which makes the spaceship slow down faster when
		 * at slower speeds.
		 */
		let speed = self.vel.length();
		let part_drag_size = DRAG_A * (speed + DRAG_B);
		let mut drag = -self.vel * part_drag_size;
		let drag_size = speed * part_drag_size;

		/*
		 * If the drag magnitude exceeds this limit, then the "instantaneous"
		 * change in velocity would result in the velocity being negated, which
		 * if the drag coefficient or delta time is sufficiently large could
		 * result in the new velocity being larger than it initially was, just on
		 * the opposite sign. This can easily spiral into being infinity and then
		 * NaN.
		 */
		let drag_limit = speed * MASS / dt.abs(); // Delta time can be negative if reversing physics
		if drag_size > drag_limit {
			drag *= drag_limit / drag_size;
		}

		force += drag;

		force / MASS
	}

	pub fn decrease_cooldown(&mut self, dt: f32) {
		self.cooldown -= dt;
	}

	// Shoots at most one bullet
	pub fn shoot(&mut self) -> bool {
		if self.cooldown <= 0.0 {
			if self.input_state.shooting() && self.ammo.is_positive() {
				self.cooldown += BULLET_INTERVAL;
				self.ammo.decrease();
				true
			} else {
				self.cooldown = 0.0;
				false
			}
		} else { false }
	}
}

impl Circle for Player {
	fn get_pos(&self) -> Vec2 { self.pos }
	fn get_radius(&self) -> f32 { RADIUS }
}
