// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::collections::{BTreeMap, BTreeSet};

use glam::Vec2;

use super::{Player, PlayerId, bullet::Bullet, powerup::{PowerupType, Effect, active_effects::{ActiveEffects, EffectInfo}}, event::move_chooser::MoveType, team::TeamId};
#[cfg(feature = "client")] use super::{World, colour::Colour, player::direction::Direction};

#[cfg(feature = "client")] use crate::utils::maths::Circle;
#[cfg(feature = "client")] use crate::playing::view::Sfx;

/*
 * Both the client and server need to know certain things that have changed
 * after updating the world. However, the exact information that's relevant to
 * them differs.
 *
 * As an optimisation, generics are used to avoid performing unnecessary
 * operations when calling `update`.
 */
pub trait Changes {
	/*
	 * For the client, prevents changes reported to old players from being
	 * visualised on players (in case if a player rapidly exits and joins).
	 *
	 * For the server, probably not needed for now but I'm adding it for good
	 * practice to clean things up when players are removed.
	 */
	fn removed_players(&mut self, _old_ids: BTreeSet<PlayerId>) {}

	fn update(&mut self, _dt: f32) {}
	fn new_player(&mut self, _id: PlayerId) {}
	fn bullet_shot(&mut self, _id: PlayerId, _player: &Player, _effects: &ActiveEffects) {}
	fn add_hit(&mut self, _bullet: Bullet, _player: &Player, _player_id: PlayerId, _killed: bool) {}
	fn block_bullet_collision(&mut self, _bullet: Bullet) {}
	fn pos_changed(&mut self, _id: PlayerId, _typ: MoveType, _prev_pos: Vec2) {}
	fn ammo_crate_collected(&mut self, _id: PlayerId, _pos: Vec2) {}
	fn clear_powerups(&mut self) {}
	fn powerup_collected(&mut self, _id: PlayerId, _player: &Player, _typ: PowerupType, _pos: Vec2) {}
	fn effect_event(&mut self, _id: PlayerId, _effect: Effect, _old_info: EffectInfo, _new_info: EffectInfo, _player_pos: Vec2, _play_sound: bool) {}
	fn flag_captured(&mut self, _pos: Vec2) {}
	fn flag_dropped(&mut self, _pos: Vec2) {}
	fn player_reset(&mut self, _id: PlayerId) {}
	fn player_team_change(&mut self, _player_id: PlayerId, _team_id: Option<TeamId>) {}
	fn items_reset(&mut self) {}
}

#[cfg(feature = "client")]
pub struct Hit {
	pub bullet: Bullet,
	pub player_id: Option<PlayerId>,
	pub player_pos: Vec2,
	pub player_vel: Vec2,
	pub player_dir: Direction,
	pub player_colour: Colour,
	pub killed: bool,
}

#[cfg(feature = "client")]
pub struct ClientChanges { // Used by the clients
	current_id: PlayerId,
	prev_shooting: BTreeSet<PlayerId>,
	collected_powerups: BTreeSet<(PlayerId, PowerupType)>,
	new_players: BTreeSet<PlayerId>,
	pub(super) update_time: f32,
	pub moved: BTreeSet<PlayerId>, // Whenever the server moves a player, which can be on death or teleportation
	pub hits: Vec<Hit>,
	pub reset: BTreeSet<PlayerId>,
	pub items_reset: bool,
	pub current_team_change: Option<TeamId>,
	pub sounds: Vec<(Sfx, Vec2, Vec2)>,
	pub powerup_sounds: Vec<(PlayerId, PowerupType, Vec2)>,
}

#[cfg(feature = "client")]
impl ClientChanges {
	pub fn new(world: &World, current_id: PlayerId) -> ClientChanges {
		ClientChanges {
			current_id,
			prev_shooting: world.players.iter().filter_map(|(id, player)| player.get_input_state().shooting().then_some(*id)).collect(),
			collected_powerups: BTreeSet::new(),
			new_players: BTreeSet::new(),
			update_time: 0.0,
			moved: BTreeSet::new(),
			hits: Vec::new(),
			reset: BTreeSet::new(),
			items_reset: false,
			current_team_change: None,
			sounds: Vec::new(),
			powerup_sounds: Vec::new(),
		}
	}

	/**
	 * Need to also check the time condition.
	 */
	pub(super) fn shot_not_predicted(&self, bullet: &Bullet) -> bool {
		bullet.get_player_owner().is_none_or(|id| id != self.current_id && !self.prev_shooting.contains(&id))
	}
}

#[cfg(feature = "client")]
impl Changes for ClientChanges {
	fn removed_players(&mut self, old_ids: BTreeSet<PlayerId>) {
		for id in &old_ids {
			self.moved.remove(id);
			self.reset.remove(id);
			self.prev_shooting.remove(id);
			self.new_players.remove(id);
		}

		self.collected_powerups.retain(|(id, _typ)| !old_ids.contains(id));

		for hit in &mut self.hits {
			if let Some(id) = hit.player_id {
				if old_ids.contains(&id) {
					hit.player_id = None;
				}
			}
		}
	}

	fn update(&mut self, dt: f32) {
		self.update_time += dt;
	}

	fn new_player(&mut self, id: PlayerId) {
		self.new_players.insert(id);
	}

	fn bullet_shot(&mut self, id: PlayerId, player: &Player, effects: &ActiveEffects) {
		// Only when the client cannot predict the next shots
		if id != self.current_id && !self.prev_shooting.contains(&id) {
			self.sounds.push((Sfx::Shot(effects.get_mul(id, Effect::Reload)), player.get_pos(), player.get_vel()));
		}
	}

	fn add_hit(&mut self, bullet: Bullet, player: &Player, player_id: PlayerId, killed: bool) {
		self.hits.push(Hit {
			bullet,
			player_id: Some(player_id),
			player_pos: player.get_pos(),
			player_vel: player.get_vel(),
			player_dir: player.get_dir(),
			player_colour: player.get_colour(),
			killed,
		});
	}

	fn block_bullet_collision(&mut self, bullet: Bullet) {
		/*
		 * Only from bullets that were shot (and the client couldn't predict the
		 * shot) and collided with a block too quickly for the smooth world to
		 * clone them.
		 */
		if self.shot_not_predicted(&bullet) && bullet.get_time() <= self.update_time {
			self.sounds.push((Sfx::BlockBullet, bullet.get_pos(), Vec2::ZERO));
		}
	}

	fn pos_changed(&mut self, id: PlayerId, typ: MoveType, prev_pos: Vec2) {
		self.moved.insert(id);

		if typ == MoveType::Teleportation {
			self.powerup_sounds.push((id, PowerupType::Teleportation, prev_pos));
		}
	}

	fn ammo_crate_collected(&mut self, _id: PlayerId, pos: Vec2) {
		self.sounds.push((Sfx::AmmoCrate, pos, Vec2::ZERO));
	}

	fn clear_powerups(&mut self) {
		self.collected_powerups.clear();
	}

	fn powerup_collected(&mut self, id: PlayerId, _player: &Player, typ: PowerupType, pos: Vec2) {
		self.collected_powerups.insert((id, typ));
		self.powerup_sounds.push((id, typ, pos));
	}

	fn effect_event(&mut self, id: PlayerId, effect: Effect, mut old_info: EffectInfo, mut new_info: EffectInfo, player_pos: Vec2, play_sound: bool) {
		if !play_sound { return; }

		/*
		 * Prevents the sound from playing whenever players join the game, which
		 * can happen if an effect zone covers the entire map.
		 */
		if self.new_players.contains(&id) { return; }

		/*
		 * If this `PlayerEffect` event is because of a powerup collection, a
		 * sound was already pushed so don't push another one.
		 */
		let typ = PowerupType::from(effect);
		if self.collected_powerups.contains(&(id, typ)) { return; }

		old_info.normalise(effect);
		new_info.normalise(effect);

		// If the effect is better
		if new_info.power > old_info.power {
			self.powerup_sounds.push((id, typ, player_pos));
		}
	}

	fn flag_captured(&mut self, pos: Vec2) {
		self.sounds.push((Sfx::FlagCapture, pos, Vec2::ZERO));
	}

	fn flag_dropped(&mut self, pos: Vec2) {
		self.sounds.push((Sfx::FlagDrop, pos, Vec2::ZERO));
	}

	fn player_reset(&mut self, id: PlayerId) {
		self.reset.insert(id);
	}

	fn items_reset(&mut self) {
		self.items_reset = true;
	}

	fn player_team_change(&mut self, player_id: PlayerId, team_id: Option<TeamId>) {
		if player_id == self.current_id {
			self.current_team_change = team_id.or(self.current_team_change);
		}
	}
}

#[derive(Default)]
pub struct ServerChanges {
	pub kills: Vec<Kill>, // Player ids duplicated if they get multiple kills in a single update
	pub(super) randomly_moved: BTreeMap<PlayerId, (MoveType, Option<TeamId>)>,
	pub(super) dead: BTreeSet<PlayerId>,
	pub(super) ammo_crates_collected: Vec<PlayerId>,
	pub(super) effects_to_apply: Vec<(PlayerId, Effect)>,
}

pub struct Kill {
	pub killer: Option<PlayerId>,
	pub killed: PlayerId,
}

impl Changes for ServerChanges {
	fn removed_players(&mut self, old_ids: BTreeSet<PlayerId>) {
		for id in &old_ids {
			self.randomly_moved.remove(id);
			self.dead.remove(id);
		}

		self.ammo_crates_collected.retain(|id| !old_ids.contains(id));
		self.effects_to_apply.retain(|(id, _)| !old_ids.contains(id));
		// Not changing self.kills as that is used before the players are eliminated
	}

	fn add_hit(&mut self, bullet: Bullet, player: &Player, player_id: PlayerId, killed: bool) {
		if killed {
			self.kills.push(Kill {
				killer: bullet.get_player_owner(),
				killed: player_id,
			});

			self.randomly_moved.insert(player_id, (MoveType::Spawn, player.get_team()));
			self.dead.insert(player_id);
		}
	}

	fn ammo_crate_collected(&mut self, id: PlayerId, _pos: Vec2) { self.ammo_crates_collected.push(id); }

	fn powerup_collected(&mut self, id: PlayerId, player: &Player, typ: PowerupType, _pos: Vec2) {
		if let Some(effect) = Effect::try_from_type(typ) {
			self.effects_to_apply.push((id, effect));
		} else if typ == PowerupType::Teleportation {
			self.randomly_moved.insert(id, (MoveType::Teleportation, player.get_team()));
		}
	}
}

pub struct IgnoreChanges;

impl Changes for IgnoreChanges {}
