// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::ops::Deref;

use serde::{Serialize, Deserialize};

use super::{Bullet, PlayerId, RADIUS, super::player::{Player, forcefield::Forcefield}};

use crate::utils::maths::{CollidingCircle, CollidingRect};
use crate::blocks::Blocks;

#[derive(Debug, Default, Clone, Serialize, Deserialize)]
pub struct Manager(Vec<Bullet>);

impl Manager {
	pub fn new() -> Manager { Manager::default() }

	pub fn clear(&mut self) {
		self.0.clear();
	}

	pub fn add(&mut self, bullet: Bullet) {
		self.0.push(bullet);
	}

	pub fn update(&mut self, dt: f32, forcefields: &[Forcefield]) {
		self.0.retain_mut(|bullet| bullet.update(dt, forcefields));
	}

	pub fn disown(&mut self, id: PlayerId) {
		for bullet in &mut self.0 {
			bullet.disown_player(id);
		}
	}

	pub fn collide_player(&mut self, player: &Player, player_id: PlayerId, friendly_fire: bool) -> Vec<Bullet> {
		let mut collided = Vec::new();
		self.0.retain(|bullet| {
			let should_remove = bullet.colliding_circle(player) && bullet.should_collide(player_id, player.get_team(), friendly_fire);
			if should_remove {
				collided.push(bullet.clone()); // Vec::extract_if would be nice but it's only in nightly Rust
			}
			!should_remove
		});
		collided
	}

	pub fn collide_blocks(&mut self, blocks: &Blocks) -> Vec<Bullet> {
		let mut collided = Vec::new();
		self.0.retain(|bullet| {
			let should_remove = blocks.iter_range(bullet.pos, RADIUS).any(|block| (bullet.pos, RADIUS).colliding_rect(block.get_rect()));
			if should_remove {
				collided.push(bullet.clone());
			}
			!should_remove
		});
		collided
	}
}

impl Deref for Manager {
	type Target = [Bullet];

	fn deref(&self) -> &Self::Target {
		self.0.as_slice()
	}
}
