// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
pub mod changes;
#[cfg(feature = "client")] mod border_renderer;
#[cfg(feature = "client")] pub mod smooth;
#[cfg(feature = "client")] mod renderer;
pub mod update;
pub mod event;
pub mod special_area;
pub mod player;
pub mod team;
pub mod bullet;
pub mod ammo_crate;
pub mod powerup;
pub mod config;
pub mod colour;
pub mod flag;

#[cfg(feature = "client")] pub use smooth::SmoothWorld;
#[cfg(feature = "client")] pub use renderer::Renderer as WorldRenderer;

use std::collections::{BTreeMap, BTreeSet};

use serde::{Serialize, Deserialize};
use glam::Vec2;
use rand::SeedableRng;
use rand_xoshiro::Xoshiro128StarStar;

use special_area::SpecialArea;
use player::{Player, PlayerId, forcefield::Forcefield};
#[cfg(feature = "client")] use player::PlayerStyle;
use team::{Team, TeamId};
use bullet::BulletManager;
use ammo_crate::AmmoCrate;
use powerup::{Powerup, Effect, active_effects::ActiveEffects};
use changes::Changes;
use config::WorldConfig;
use flag::{Flag, FlagState};

use crate::utils::maths::Circle;

pub type WorldRng = Xoshiro128StarStar;

const UPDATES_PER_SECOND: f32 = 240.0;
pub const DELTA_TIME: f32 = 1.0 / UPDATES_PER_SECOND;

// At most 60 seconds to avoid DoS attacks clogging up queues
pub const MAX_STEPS: usize = (60.0 * UPDATES_PER_SECOND) as usize;

// Arbitrary limits used by the client to avoid application layer denial of service attacks
#[cfg(feature = "client")] pub const MAX_PLAYERS: usize = 1024;
#[cfg(feature = "client")] pub const MAX_TEAMS: usize = 1024;
#[cfg(feature = "client")] pub const MAX_BULLETS: usize = 16384;
#[cfg(feature = "client")] pub const MAX_SPECIAL_AREAS: usize = 4096;
const MAX_AMMO_CRATES: usize = 512;
const MAX_POWERUPS: usize = 512;

/**
 * NOTE: All code that is run on both the client and server **must be
 * deterministic**. If not, it's possible that the client's world state gets out
 * of sync of the server state. This could result in, say, a player being at one
 * part of the world on the client and at another part of the world on the
 * server.
 *
 * This means that **anything** that either *is* or *might be* non-deterministic
 * cannot be used.
 *
 * This includes the obvious like calls to `rand::random` but also includes less
 * obvious behaviour like iterating through `HashMap`s or `HashSet`s as the
 * iteration order depends on the random state. In the latter example, to
 * eliminate the probability of accidentally iterating through a `HashMap` or
 * `HashSet`, they **should not** be used in this code.
 *
 * Another thing to watch out for is the use of certain mathematical functions
 * which the Rust documentation states have "unspecified precision" like `sin`
 * and `exp`. Deterministic functions in a crate like `libm` should be used
 * instead. The `glam` crate has been configured to use `libm`.
 *
 * Note that these strict requirements don't exist for code that just runs on
 * the server (like generating events input to the world) or the client (like
 * rendering).
 */
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct World {
	config: WorldConfig,
	players: BTreeMap<PlayerId, Player>,
	teams: Box<[Team]>,
	bullets: BulletManager,
	ammo_crates: Vec<AmmoCrate>,
	powerups: Vec<Powerup>,
	effects: ActiveEffects,
	rng: WorldRng,
	special_areas: Box<[SpecialArea]>,
	flags: Box<[Flag]>,
}

impl World {
	pub fn new(config: WorldConfig, special_areas: Box<[SpecialArea]>, teams: Box<[Team]>, flags: Box<[Flag]>) -> World {
		World {
			config,
			players: BTreeMap::new(),
			teams,
			bullets: BulletManager::new(),
			ammo_crates: Vec::new(),
			powerups: Vec::new(),
			effects: ActiveEffects::new(),
			rng: WorldRng::seed_from_u64(rand::random::<u64>()),
			special_areas,
			flags,
		}
	}

	/** Validation that just needs to be performed once. */
	#[cfg(feature = "client")]
	pub fn validate_init(&self) -> Result<(), &'static str> {
		if self.special_areas.len() > MAX_SPECIAL_AREAS { return Err("too many special areas in world"); }
		if self.teams.len() > MAX_TEAMS { return Err("too many teams"); }

		Ok(())
	}

	/** Validation that needs to be performed every update. */
	#[cfg(feature = "client")]
	pub fn validate(&self) -> Result<(), &'static str> {
		if !self.config.size.cmpge(Vec2::splat(0.0)).all() { return Err("invalid world size"); } // Need to do ! >= rather than <, as the latter allows a world size of NaN
		if self.players.len() > MAX_PLAYERS { return Err("too many players in world"); }
		if self.bullets.len() > MAX_BULLETS { return Err("too many bullets in world"); }
		if self.ammo_crates.len() > MAX_AMMO_CRATES { return Err("too many ammo crates in world"); }
		if self.powerups.len() > MAX_POWERUPS { return Err("too many powerups in world"); }
		self.effects.validate()
	}

	pub fn player_count(&self) -> usize { self.players.len() }
	pub fn get_players(&self) -> &BTreeMap<PlayerId, Player> { &self.players }
	#[cfg(feature = "client")] fn get_ammo_crates(&self) -> &[AmmoCrate] { &self.ammo_crates }
	#[cfg(feature = "client")] pub fn get_powerups(&self) -> &[Powerup] { &self.powerups }

	fn get_forcefields(&self) -> Vec<Forcefield> {
		self.effects
			.player_iter_with_effect(Effect::Forcefield)
			.map(|(player_id, info)| {
				let player = &self.players[&player_id];
				Forcefield { player_id, team_id: player.get_team(), pos: player.get_pos(), strength: info.power }
			})
			.collect()
	}

	pub fn has_player(&self, id: PlayerId) -> bool { self.players.contains_key(&id) }
	pub fn player_id_iter(&self) -> impl Iterator<Item = PlayerId> + '_ { self.players.keys().copied() }
	pub fn team_id_iter(&self) -> impl Iterator<Item = TeamId> + '_ { (0..self.teams.len()).map(TeamId::from_index) }

	fn remove_players(&mut self, ids: BTreeSet<PlayerId>, changes: &mut impl Changes) {
		if ids.is_empty() { return; }

		for flag in &mut self.flags {
			if let FlagState::Following(id, animation) = flag.get_state() {
				if ids.contains(id) {
					flag.set_state(FlagState::Fixed(self.players.get(id).map_or(Vec2::NAN, Circle::get_pos) + animation.get_dpos(0.0)), changes, |_| None /* Falls back to the same position */);
				}
			}
		}

		for &id in &ids {
			self.players.remove(&id);
			self.effects.remove_player(id);
			self.bullets.disown(id);
		}

		changes.removed_players(ids);
	}

	#[cfg(feature = "client")]
	pub fn get_player_style(&self, id: PlayerId) -> Option<PlayerStyle> {
		self.players.get(&id).map(Player::get_style)
	}

	/**
	 * Returns Some(id) of the team the next player should join if teams are
	 * enabled, otherwise None if there are no teams.
	 */
	pub fn next_team(&self) -> Option<TeamId> {
		self.team_freqs().into_iter().enumerate().min_by_key(|&(_id, freq)| freq).map(|(id, _freq)| TeamId::from_index(id))
	}

	pub fn has_team(&self, id: TeamId) -> bool {
		id.index() < self.teams.len()
	}

	pub fn team_count(&self) -> usize {
		self.teams.len()
	}

	pub fn has_teams(&self) -> bool {
		!self.teams.is_empty()
	}

	pub fn team_freqs(&self) -> Vec<usize> {
		let mut freqs = vec![0usize; self.teams.len()];
		for id in self.players.values().filter_map(Player::get_team) {
			if let Some(f) = freqs.get_mut(id.index()) {
				*f += 1;
			}
		}
		freqs
	}

	pub fn get_team(&self, id: TeamId) -> Option<&Team> {
		self.teams.get(id.index())
	}

	pub fn get_player_team(&self, id: PlayerId) -> Option<TeamId> {
		self.players.get(&id).and_then(Player::get_team)
	}

	pub fn teams_to_players(&self, teams: BTreeSet<TeamId>) -> BTreeSet<PlayerId> {
		self.players
			.iter()
			.filter_map(|(&id, player)| player.get_team().is_some_and(|team| teams.contains(&team)).then_some(id))
			.collect()
	}
}
