// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::{array, path::Path, time::Duration, rc::{Rc, Weak}, cell::RefCell, collections::{BTreeMap, btree_map::Entry}};

use kira::{
	AudioManager, AudioManagerSettings, DefaultBackend, Tween, StartTime, Easing,
	Decibels, listener::ListenerHandle, sound::{PlaybackState,
	static_sound::{StaticSoundData, StaticSoundHandle}},
	track::{SpatialTrackBuilder, SpatialTrackHandle},
};
use glam::{Vec2, Vec3, Quat};
use strum_macros::EnumCount as EnumCountMacro;
use strum::EnumCount;
use num_enum::TryFromPrimitive;

use crate::game::{config::Config, filesystem::{Filesystem, FsBase}};
use crate::world::powerup::PowerupType;
use crate::net::message::GameStateTransition;
use crate::utils::maths::{self, glam_fix::Fix};

#[repr(usize)]
#[derive(EnumCountMacro, TryFromPrimitive)]
enum SfxVariant {
	Shot,
	Hit,
	Kill,
	BlockBullet,
	AmmoCrate,
	Speed,
	Reload,
	Regeneration,
	Damage,
	Forcefield,
	Teleportation,
	Thrust,
	BlockPlayer,
	GameStart,
	GameEnd,
	GameWin,
	FlagCapture,
	FlagDrop,
}

impl SfxVariant {
	fn name(self) -> &'static str {
		match self {
			SfxVariant::Shot => "shot",
			SfxVariant::Hit => "hit",
			SfxVariant::Kill => "kill",
			SfxVariant::BlockBullet => "block_bullet",
			SfxVariant::AmmoCrate => "ammo_crate",
			SfxVariant::Speed => "speed",
			SfxVariant::Reload => "reload",
			SfxVariant::Regeneration => "regen",
			SfxVariant::Damage => "damage",
			SfxVariant::Forcefield => "forcefield",
			SfxVariant::Teleportation => "teleportation",
			SfxVariant::Thrust => "thrust",
			SfxVariant::BlockPlayer => "block_player",
			SfxVariant::GameStart => "game_start",
			SfxVariant::GameEnd => "game_end",
			SfxVariant::GameWin => "game_win",
			SfxVariant::FlagCapture => "flag_capture",
			SfxVariant::FlagDrop => "flag_drop",
		}
	}
}

pub enum Sfx {
	Shot(f32),
	Hit,
	Kill,
	BlockBullet,
	AmmoCrate,
	Powerup(PowerupType), // For powerups and effect zones
	GameStateTransition(GameStateTransition),
	FlagCapture,
	FlagDrop,
}

impl Sfx {
	fn get_speed(&self) -> f32 {
		if let Sfx::Shot(reload) = self {
			if *reload > 1.0 { 3.0 - 4.0 / (*reload + 1.0) } else { 1.0 }
		} else {
			1.0
		}
	}

	fn index(&self) -> usize {
		(match self {
			Sfx::Shot(_) => SfxVariant::Shot,
			Sfx::Hit => SfxVariant::Hit,
			Sfx::Kill => SfxVariant::Kill,
			Sfx::BlockBullet => SfxVariant::BlockBullet,
			Sfx::AmmoCrate => SfxVariant::AmmoCrate,
			Sfx::Powerup(PowerupType::Speed) => SfxVariant::Speed,
			Sfx::Powerup(PowerupType::Reload) => SfxVariant::Reload,
			Sfx::Powerup(PowerupType::Health) => SfxVariant::Regeneration,
			Sfx::Powerup(PowerupType::Damage) => SfxVariant::Damage,
			Sfx::Powerup(PowerupType::Forcefield) => SfxVariant::Forcefield,
			Sfx::Powerup(PowerupType::Teleportation) => SfxVariant::Teleportation,
			Sfx::GameStateTransition(GameStateTransition::Start) => SfxVariant::GameStart,
			Sfx::GameStateTransition(GameStateTransition::End) => SfxVariant::GameEnd,
			Sfx::GameStateTransition(GameStateTransition::Win) => SfxVariant::GameWin,
			Sfx::FlagCapture => SfxVariant::FlagCapture,
			Sfx::FlagDrop => SfxVariant::FlagDrop,
		}) as usize
	}
}

#[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub enum LoopingSfx {
	Thrust,
	BlockPlayer,
}

impl LoopingSfx {
	fn index(self) -> usize {
		(match self {
			LoopingSfx::Thrust => SfxVariant::Thrust,
			LoopingSfx::BlockPlayer => SfxVariant::BlockPlayer,
		}) as usize
	}

	fn pause_time(self) -> f32 {
		match self {
			LoopingSfx::Thrust => 0.375,
			LoopingSfx::BlockPlayer => 0.5,
		}
	}
}

pub struct Sounds {
	pos: Vec2,
	vel: Vec2,

	config: Rc<RefCell<Config>>,
	manager: AudioManager<DefaultBackend>,
	listener: ListenerHandle,
	spatial_tracks: Vec<(SpatialTrackHandle, StaticSoundHandle)>, // Could call `persist_until_sounds_finish` but it doesn't seem to perform as well when there are many ongoing sound effects
	looping_tracks: Vec<Weak<RefCell<LoopingTrackInner>>>,

	/*
	 * Spatial audio used when implementing non-spatial audio, but the track's
	 * position is at (0, 0, 0) and the listener's position is at (0, 0, z).
	 *
	 * This is to reproduce the exact amplitude if the sound was spatial but
	 * centred at the camera's x and y coordinates.
	 */
	non_spatial_listener: ListenerHandle,
	non_spatial_track: SpatialTrackHandle,

	sound_effects: [StaticSoundData; SfxVariant::COUNT],
}

pub struct LoopingTrack(Option<Rc<RefCell<LoopingTrackInner>>>);

struct LoopingTrackInner {
	pos: Vec2,
	vel: Vec2,
	track: SpatialTrackHandle,
	sounds: BTreeMap<LoopingSfx, LoopingTrackSound>,
}

struct LoopingTrackSound {
	playing: bool,

	/**
	 * The code that uses this exists to work around an annoying problem in
	 * `kira` where the looping sound initially plays for a short amount of time
	 * when it's created even if playing is set to false.
	 */
	muted: bool,

	/**
	 * None = The state when adding a sound for the first time.
	 * Some(...) = The state after `Sounds::update` is called.
	 * Some(Ok) = Sound successfully created.
	 * Some(Err) = Sound failed to be created.
	 */
	handle: Option<Result<StaticSoundHandle, ()>>,
}

impl LoopingTrack {
	/**
	 * Cheaply creates a `LoopingTrack` that doesn't play any sound.
	 */
	pub fn nothing() -> LoopingTrack {
		LoopingTrack(None)
	}

	pub fn update(&self, pos: Vec2, vel: Vec2, sounds: &[(LoopingSfx, bool)]) {
		let Some(inner) = &self.0 else { return; };
		let mut inner = inner.borrow_mut();

		inner.pos = pos;
		inner.vel = vel;

		for (sfx, playing) in sounds {
			match inner.sounds.entry(*sfx) {
				Entry::Vacant(e) => _ = e.insert(LoopingTrackSound { playing: *playing, muted: true, handle: None }),
				Entry::Occupied(mut e) => {
					let sound = e.get_mut();

					if let Some(Ok(handle)) = &mut sound.handle {
						if *playing {
							if sound.muted {
								handle.set_volume(Decibels::IDENTITY, new_tween(0.0));
								sound.muted = false;
							}
							handle.resume(new_tween(RESUME_TIME));
						} else if !playing {
							handle.pause(new_tween(sfx.pause_time()));
						}
					}

					sound.playing = *playing;
				},
			}
		}
	}
}

const SOUND_SPEED: f32 = 500.0;
const RESUME_TIME: f32 = 0.03125;
const TOGGLE_MUTE_TIME: f32 = 0.25;
const MAX_PLAYBACK_SPEED: f32 = 32.0;

impl Sounds {
	pub fn new(fs: &Filesystem, config: Rc<RefCell<Config>>) -> Sounds {
		let sound_effects = array::from_fn(|i| {
			let name = SfxVariant::try_from(i).unwrap().name();
			let path = fs.get(FsBase::Static, Path::new(&format!("sounds/{name}.wav")));
			StaticSoundData::from_file(path).unwrap()
		});

		let mut settings = AudioManagerSettings::default();
		settings.capacities.sub_track_capacity = 256;
		let mut manager = AudioManager::new(settings).unwrap();
		manager.main_track().set_volume(config.borrow().sound.get_volume(), new_tween(0.0));
		let listener = manager.add_listener(Vec3::ZERO, Quat::IDENTITY).unwrap();
		let non_spatial_listener = manager.add_listener(Vec3::ZERO, Quat::IDENTITY).unwrap();
		let non_spatial_track = manager.add_spatial_sub_track(&non_spatial_listener, Vec3::ZERO, spatial_track_builder()).unwrap();

		Sounds {
			pos: Vec2::ZERO,
			vel: Vec2::ZERO,
			config,
			manager,
			listener,
			non_spatial_listener,
			non_spatial_track,
			spatial_tracks: Vec::new(),
			looping_tracks: Vec::new(),
			sound_effects,
		}
	}

	pub fn set_muted(&mut self, muted: bool) {
		self.manager.main_track().set_volume(if muted { Decibels::SILENCE } else { self.config.borrow().sound.get_volume() }, new_tween(TOGGLE_MUTE_TIME));
	}

	pub fn play(&mut self, sfx: Sfx) {
		let sound = self.get_static_sound(sfx, 1.0);
		let _ = self.non_spatial_track.play(sound);
	}

	pub fn play_spatial(&mut self, sfx: Sfx, pos: Vec2, vel: Vec2) {
		let doppler = Sounds::get_doppler(self.pos, self.vel, pos, vel);
		let sound = self.get_static_sound(sfx, doppler);

		let Ok(mut track) = self.manager.add_spatial_sub_track(&self.listener, into_3d(pos), spatial_track_builder()) else { return; };
		let Ok(sound) = track.play(sound) else { return; };
		self.spatial_tracks.push((track, sound));
	}

	fn get_doppler(l_pos: Vec2, l_vel: Vec2, s_pos: Vec2, s_vel: Vec2) -> f32 {
		/*
		 * Don't want the Doppler effect to be 3D as it feels weird when the zoom
		 * changes.
		 */
		let dir = (l_pos - s_pos).normalise_or(Vec2::ZERO);
		let doppler = (SOUND_SPEED - dir.dot(l_vel)) / (SOUND_SPEED - dir.dot(s_vel));

		if doppler < 0.0 || doppler.is_nan() { // Supersonic speeds
			f32::INFINITY
		} else {
			doppler
		}
	}

	pub fn get_looping_track(&mut self, pos: Vec2, vel: Vec2) -> LoopingTrack {
		LoopingTrack(self.get_looping_track_inner(pos, vel).map(|looping| {
			let ret = Rc::new(RefCell::new(looping));
			self.looping_tracks.push(Rc::downgrade(&ret));
			ret
		}))
	}

	fn get_looping_track_inner(&mut self, pos: Vec2, vel: Vec2) -> Option<LoopingTrackInner> {
		let Ok(track) = self.manager.add_spatial_sub_track(&self.listener, into_3d(pos), spatial_track_builder()) else { return None; };
		Some(LoopingTrackInner { pos, vel, track, sounds: BTreeMap::new() })
	}

	fn get_static_sound(&self, sfx: Sfx, speed: f32) -> StaticSoundData {
		self.sound_effects[sfx.index()].playback_rate((speed * sfx.get_speed()).min(MAX_PLAYBACK_SPEED) as f64)
	}

	pub fn update(&mut self, pos: Vec2, zoom: f32, jolted: bool, dt: f32) {
		let tween = new_tween(dt);
		self.vel = if jolted { Vec2::ZERO } else { (pos - self.pos) / maths::limit_dt(dt) };
		self.pos = pos;

		let z = zoom / 2.0;

		self.listener.set_position(pos.extend(z), tween);
		self.non_spatial_listener.set_position(Vec2::ZERO.extend(z), tween);
		self.spatial_tracks.retain(|(_, sound)| sound.state() == PlaybackState::Playing);
		self.looping_tracks.retain(|looping| {
			let Some(looping) = looping.upgrade() else { return false; };
			let inner = &mut *looping.borrow_mut();

			inner.track.set_position(into_3d(inner.pos), tween);
			let doppler = Sounds::get_doppler(self.pos, self.vel, inner.pos, inner.vel);
			let speed = doppler.abs().min(MAX_PLAYBACK_SPEED) as f64;

			for (sfx, sound) in &mut inner.sounds {
				let Ok(handle) = sound.handle.get_or_insert_with(|| {
					/*
					 * Initially muted so all sounds start off as paused and don't
					 * play anything at first.
					 *
					 * Without this, things most likely will be fine but it might be
					 * possible that the sound plays a very short amount initially.
					 */
					let s = self.sound_effects[sfx.index()].loop_region(..).volume(Decibels::SILENCE);
					let mut handle = inner.track.play(s).map_err(|_| ());
					if let Ok(handle) = &mut handle {
						handle.pause(new_tween(0.0));
						handle.seek_to(0.0);

						// If the new sound is playing, have it fade in
						if sound.playing {
							handle.set_volume(Decibels::IDENTITY, new_tween(0.0));
							sound.muted = false;
							handle.resume(new_tween(RESUME_TIME));
						}
					}
					handle
				}) else { continue; };

				handle.set_playback_rate(speed, tween);
			}

			true
		});
	}
}

fn new_tween(time: f32) -> Tween {
	Tween { start_time: StartTime::Immediate, duration: Duration::from_secs_f32(time), easing: Easing::Linear }
}

fn spatial_track_builder() -> SpatialTrackBuilder {
	SpatialTrackBuilder::new()
		.distances((0.0, 500.0))
		.attenuation_function(Some(Easing::InPowi(2)))
}

fn into_3d(pos: Vec2) -> Vec3 {
	pos.extend(0.0)
}
