// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use glam::Vec2;
use egui::{Context, LayerId, Order, Id, FontId, Rounding, Color32};

use crate::world::{colour::Colour, player::PlayerStyle};

pub(super) const FONT_SIZE: f32 = 12.0;

pub struct NameAbovePlayer {
	pos: Vec2, // Logical pixels
	style: PlayerStyle,
}

impl NameAbovePlayer {
	pub fn new(pos: Vec2, style: PlayerStyle) -> NameAbovePlayer {
		NameAbovePlayer { pos, style }
	}

	pub fn render(&self, ctx: &Context) {
		let painter = ctx.layer_painter(LayerId { order: Order::Background, id: Id::NULL });
		let colour = Colour::name_colour_normal(self.style.colour.name_colour_dark());
		let galley = painter.layout_no_wrap(process(self.style.name.as_ref()), FontId::proportional(FONT_SIZE), colour);

		// Renders a rectangle under it to make it easier to read
		let mut bg_rect = galley.rect;
		bg_rect.set_center(self.pos.to_array().into());
		painter.rect_filled(bg_rect.expand(3.0), Rounding::ZERO, Color32::from_black_alpha(0x7f));

		// Renders the text
		let size = galley.size();
		let text_pos = self.pos - Vec2::new(size.x / 2.0, size.y / 2.0);
		painter.galley(text_pos.to_array().into(), galley, colour);
	}
}

/**
 * Does two jobs.
 *
 * The first is truncating the names if they're too long to prevent them from
 * being too long and looking weird.
 *
 * The second is to remove any newlines in the edge case that a user adds a
 * newline (probably to try and break things).
 */
pub(super) fn process(name: &str) -> String {
	const MAX_LEN: usize = 20;

	let mut ret = String::with_capacity(name.len().min(MAX_LEN + 1)); // Works in most cases

	/*
	 * Limiting based on the number of characters, not bytes.
	 *
	 * This is important because the purpose of this function is to prevent the
	 * name from being too long to avoid it looking weird when being rendered.
	 *
	 * I'm not interested in the length in bytes, and using bytes would mean
	 * users who use names in other non-ASCII languages would be disadvantaged,
	 * having their name truncated earlier.
	 */
	let mut count = 0;
	for ch in name.chars() {
		if ch == '\n' { continue; }

		if count >= MAX_LEN {
			ret.push('…');
			break
		}

		ret.push(ch);
		count += 1;
	}

	ret
}
