// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
mod ammo_status;
mod scores;
mod timer;
pub mod health_bar;
pub mod text_io;
pub mod names;

use glium::{Display, Frame, DrawParameters};
use glutin::surface::WindowSurface;
use winit::event::{KeyEvent, ElementState};
use glam::{UVec2, Vec2};
use egui::Context;

use ammo_status::AmmoStatus;
use text_io::TextIo;

use super::{Camera, HealthBarRenderer, TimeArgs, NameAbovePlayer};

use crate::game::{Game, filesystem::Filesystem, config::action::ActionEvent};
use crate::world::{World, SmoothWorld};
use crate::net::client::Client;

pub struct Hud {
	ammo_status: AmmoStatus,
	health_bars: HealthBarRenderer,
	text_io: TextIo,
	winsize: UVec2,
	scale_factor: f32,
}

impl Hud {
	pub fn new(display: &Display<WindowSurface>, fs: &Filesystem, winsize: UVec2, scale_factor: f32) -> Hud {
		Hud {
			ammo_status: AmmoStatus::new(display, fs),
			health_bars: HealthBarRenderer::new(display, fs),
			text_io: TextIo::new(),
			winsize,
			scale_factor,
		}
	}

	pub fn reset(&mut self, winsize: UVec2, scale_factor: f32) {
		self.resize_event(winsize);
		self.scale_factor_event(scale_factor);
		self.text_io = TextIo::new();
	}

	pub fn get_text_io(&mut self) -> &mut TextIo { &mut self.text_io }
	pub fn text_input_open(&self) -> bool { self.text_io.open() }

	pub fn action(&mut self, action: ActionEvent) {
		self.text_io.action(action);
	}

	pub fn key_event(&mut self, event: &KeyEvent) {
		if !self.text_io.open() { return; }

		if let KeyEvent { text: Some(text), state: ElementState::Pressed, .. } = &event {
			self.text_io.push_current(text);
		}
	}

	pub fn resize_event(&mut self, winsize: UVec2) { self.winsize = winsize; }
	pub fn scale_factor_event(&mut self, scale_factor: f32) { self.scale_factor = scale_factor; }

	pub fn push_health_bars(&mut self, smooth_world: &SmoothWorld, camera: &Camera, time_diff: f32) -> Vec<NameAbovePlayer> {
		let lwinsize = self.get_lwinsize();
		smooth_world.push_health_bars(&mut self.health_bars, lwinsize, camera, time_diff)
	}

	#[allow(clippy::too_many_arguments)]
	pub fn render(&mut self, time: &TimeArgs, game: &mut Game, frame: &mut Frame, params: &mut DrawParameters, world: &World, smooth_world: &SmoothWorld, client: &mut Client, names: &[NameAbovePlayer]) {
		let lwinsize = self.get_lwinsize();

		self.health_bars.render(&game.display, frame, params);

		let ammo_count = smooth_world.followed_player().map(|(_, player)| player.get_ammo_string());
		let render_ammo_status = ammo_count.is_some();

		Hud::render_gui(game, frame, |ctx| {
			for name in names {
				name.render(ctx);
			}
		});

		if let Some(count) = ammo_count {
			self.ammo_status.render(frame, params, lwinsize);
			self.ammo_status.set_text(count);
		}

		Hud::render_gui(game, frame, |ctx| {
			scores::render(ctx, client.get_scores(), smooth_world.followed_player_id(), smooth_world.followed_team(), world);
			if let Some(time) = client.get_time(smooth_world.get_total_time_shift() + time.time_diff) {
				timer::render(ctx, time);
			}
			self.text_io.render(time.dt, ctx, lwinsize, smooth_world.player_count(), client);
			if render_ammo_status {
				self.ammo_status.render_text(ctx, lwinsize);
			}
		});
	}

	fn render_gui(game: &mut Game, frame: &mut Frame, mut f: impl FnMut(&Context)) {
		game.gui.update(&game.window, |ctx| {
			/*
			 * Temporarily sets the animation time to zero so that the ammo status
			 * window appears immediately.
			 *
			 * Also don't want labels to be selectable for any initial events
			 * (haven't observed this but I want to be safe).
			 */
			let mut prev_animation_time = 0.0;
			let mut prev_selectable_labels = false;
			ctx.style_mut(|style| {
				prev_animation_time = style.animation_time;
				prev_selectable_labels = style.interaction.selectable_labels;
				style.animation_time = 0.0;
				style.interaction.selectable_labels = false;
			});

			f(ctx);

			ctx.style_mut(|style| {
				style.animation_time = prev_animation_time;
				style.interaction.selectable_labels = prev_selectable_labels;
			});
		});

		game.gui.render(&game.display, frame);
	}

	fn get_lwinsize(&self) -> Vec2 {
		self.winsize.as_vec2() / self.scale_factor
	}
}
