// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use glam::{Vec3, Vec4};

use crate::utils::maths::decay::Decay;

#[derive(Clone, Copy)]
pub struct Temperature(f32);

/**
 * The colour of a thrust particle is based on its temperature, a
 * floating-point value which ranges between 1 – 4. This temperature is similar
 * to what you get from black-body radiation, and more precisely is a linear
 * gradient with these points:
 * 	T(1) = (0, 0, 0, 0) (fully transparent black)
 * 	T(2) = (1, 0, 0, 0.5) (half transparent red)
 * 	T(3) = (1, 1, 0, 0.875) (mostly opaque yellow)
 * 	T(4) = (1, 1, 1, 1) (white)
 */
impl Temperature {
	pub fn new(temp: f32) -> Temperature { Temperature(temp) }
	pub fn visible(self) -> bool { self.0 >= 1.0 }

	/**
	 * Exponentially decays the temperature, modelled after Newton's law of
	 * cooling.
	 */
	pub fn cool(&mut self, rate: f32, dt: f32) {
		self.0.decay(rate, dt);
	}

	pub fn get_colour_rgba(self) -> Vec4 {
		match self.0 {
			..=1.0 => Vec4::ZERO,
			1.0..=2.0 => Vec4::new(self.0 - 1.0, 0.0, 0.0, (self.0 - 1.0) * 0.5),
			2.0..=3.0 => Vec4::new(1.0, self.0 - 2.0, 0.0, self.0 * 0.375 - 0.25),
			3.0..=4.0 => Vec4::new(1.0, 1.0, self.0 - 3.0, 1.0),
			_ => Vec4::ONE, // Not checking for NaNs
		}
	}

	pub fn get_colour_rgb(self) -> Vec3 {
		let rgba = self.get_colour_rgba();
		Vec3::new(rgba.x * rgba.w, rgba.y * rgba.w, rgba.z * rgba.w)
	}
}

