// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::f32::consts::TAU;

use glam::{Vec2, Vec4};

use super::super::Particle;

use crate::world::powerup::RADIUS;

pub struct Damage {
	centre: Vec2,
	max: Vec2,
	size: f32,
	green: f32,
	blue: f32,
	phase: f32,
	pos: f32,
	freq: f32,
	time: f32,
	max_time: f32,
}

impl Damage {
	pub fn new(pos: Vec2) -> Damage {
		let freq = (rand::random::<f32>() * 4.0 + 6.0).sqrt();
		let centre = pos + Vec2::from_angle(rand::random::<f32>() * TAU) * rand::random::<f32>() * RADIUS;
		let green = rand::random::<f32>() * 0.125 + 0.125;
		Damage {
			centre,
			max: centre + Vec2::from_angle(rand::random::<f32>() * TAU) * (rand::random::<f32>() * rand::random::<f32>() + 1.0) * 0.4375,
			size: (rand::random::<f32>() + 1.0) * 0.03125,
			green,
			blue: green * (rand::random::<f32>() * 0.25 + 0.625),
			pos: 0.0,
			phase: rand::random::<f32>() * TAU,
			freq,
			time: 0.0,
			max_time: rand::random::<f32>() * 2.0 + 6.0,
		}
	}
}

impl Particle for Damage {
	fn get_pos(&self) -> Vec2 {
		self.centre.lerp(self.max, self.pos)
	}

	fn get_size(&self) -> f32 { self.size }

	fn get_colour(&self) -> Vec4 {
		let t = self.time / self.max_time;
		Vec4::new(0.5, self.green, self.blue, 4.0 * t * (1.0 - t) * (self.pos * 4.0).powi(2).min(1.0) /* Decreases the opacity when closer to the centre */)
	}

	fn update(&mut self, dt: f32) -> bool {
		self.pos = (self.time * self.freq + self.phase).sin();
		self.time += dt;
		self.time <= self.max_time
	}
}
