// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::f32::consts::TAU;

use glam::{Vec2, Vec3, Vec4, swizzles::Vec4Swizzles};

use super::{Particle, temperature::Temperature};

use crate::world::{colour::Colour, player::MIN_DIST_TO_EDGE, changes::Hit};

pub struct Explosion {
	pos: Vec2,
	vel: Vec2,
	size: f32,
	temp: Temperature,
	colour: Colour,
}

impl Explosion {
	pub fn new(hit: &Hit) -> Explosion {
		Explosion {
			pos: hit.player_pos + rand::random::<Vec2>() * MIN_DIST_TO_EDGE * 2.0 - MIN_DIST_TO_EDGE,
			vel: hit.player_vel + Vec2::from_angle(rand::random::<f32>() * TAU) * rand::random::<f32>() * 3.75,
			size: rand::random::<f32>() * 0.25 + 0.125,
			temp: Temperature::new(rand::random::<f32>() * 2.0 + 2.0),
			colour: hit.player_colour,
		}
	}
}

impl Particle for Explosion {
	fn get_pos(&self) -> Vec2 { self.pos }
	fn get_size(&self) -> f32 { self.size }

	fn get_colour(&self) -> Vec4 {
		let temp_rgba = self.temp.get_colour_rgba();
		(Vec3::from(self.colour) + temp_rgba.xyz() * temp_rgba.w).extend(temp_rgba.w)
	}

	fn update(&mut self, dt: f32) -> bool {
		self.pos += self.vel * dt;
		self.temp.cool(3.75, dt);
		self.temp.visible()
	}
}
