// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
mod decorations;
mod hud;
mod camera;
mod sounds;

pub use camera::Camera;
pub use hud::{health_bar::{HealthBar, HealthBarRenderer}, names::NameAbovePlayer};
pub use sounds::{Sounds, Sfx, LoopingTrack, LoopingSfx};

use std::{rc::Rc, cell::RefCell};

use glium::{Frame, DrawParameters};
use winit::{event::{WindowEvent, MouseScrollDelta}, dpi::PhysicalSize};
use glam::{UVec2, Vec2};

use decorations::Decorations;
use camera::DecayMode;
use hud::Hud;

use super::TimeArgs;
use super::model::Model;
use super::controller::Controller;

use crate::game::{Game, config::{Config, action::{Action, ActionEvent, ActionState}}};
use crate::world::{SmoothWorld, WorldRenderer, changes::Hit as ChangesHit};
use crate::utils::blending::ALPHA_BLENDING;

pub(super) struct View<'a> {
	pub smooth_world: SmoothWorld,
	pub camera: Camera,
	pub sounds: Sounds,
	params: DrawParameters<'a>,
	decorations: Decorations,
	world_renderer: WorldRenderer,
	hud: Hud,
	render_hud: bool,
}

impl<'a> View<'a> {
	pub fn new(config: Rc<RefCell<Config>>, game: &mut Game) -> View<'a> {
		let smooth_world = SmoothWorld::new();
		let winsize = { let PhysicalSize { width, height } = game.window.inner_size(); UVec2::new(width, height) };
		let camera = Camera::new(winsize);

		let player_gfx = game.player_gfx.get(&game.display, &game.fs);
		let world_renderer = WorldRenderer::new(&game.display, &game.fs, Rc::clone(&game.config), Rc::clone(&player_gfx));
		let decorations = Decorations::new(config, &game.display, &game.fs, player_gfx);

		View {
			smooth_world,
			camera,
			sounds: Sounds::new(&game.fs, Rc::clone(&game.config)),
			params: DrawParameters {
				blend: ALPHA_BLENDING,
				.. DrawParameters::default()
			},
			decorations,
			world_renderer,
			hud: Hud::new(&game.display, &game.fs, winsize, game.window.scale_factor() as f32),
			render_hud: true,
		}
	}

	pub fn reset(&mut self, game: &Game, model: &Model) {
		let size = {
			let size = game.window.inner_size();
			UVec2::new(size.width, size.height)
		};

		self.hud.reset(size, game.window.scale_factor() as f32);
		self.smooth_world.reset(model, &mut self.sounds);
		if let Some((_, player)) = self.smooth_world.followed_player() {
			self.camera.reset(player.get_pos(0.0), player.get_vel(), size);
		} else {
			self.camera.reset(Vec2::ZERO, Vec2::ZERO, size);
		}
		self.update_sounds(0.0);
		self.decorations.reset(&game.display, size, &self.camera);
		self.world_renderer.reset(&game.display, model);
		self.render_hud = true;
	}

	pub fn text_input_action(&mut self, action: ActionEvent) {
		if self.render_hud {
			/*
			 * Doesn't do this if the HUD is hidden, as the player will no longer
			 * be able to move with no visible feedback to why that's the case.
			 */
			self.hud.action(action);
		}
	}

	pub fn action(&mut self, action: ActionEvent) {
		self.camera.action(action);

		if matches!(action, ActionEvent::Action(Action::ToggleHud, ActionState::Pressed)) {
			self.render_hud = !self.render_hud;
		}

		if let Some(new_camera_pos) = self.smooth_world.action(action) {
			self.camera.move_to(new_camera_pos);
		}
	}

	pub fn following_player(&self) -> bool {
		self.smooth_world.followed_player().is_some()
	}

	pub fn get_hud(&self) -> &Hud { &self.hud }
	pub fn get_hud_mut(&mut self) -> &mut Hud { &mut self.hud }

	pub fn window_event(&mut self, event: &WindowEvent) {
		match event {
			WindowEvent::Resized(size) => {
				let winsize = UVec2::new(size.width, size.height);
				self.camera.resize_event(winsize);
				self.hud.resize_event(winsize);
				self.decorations.resize_event(winsize);
			}
			WindowEvent::ScaleFactorChanged { scale_factor, .. } => self.hud.scale_factor_event(*scale_factor as f32),
			WindowEvent::MouseWheel { delta: MouseScrollDelta::LineDelta(_, dy), .. } => self.camera.mouse_scroll_event(*dy),
			WindowEvent::MouseWheel { delta: MouseScrollDelta::PixelDelta(pos), .. } => self.camera.mouse_scroll_event(pos.y as f32 * 0.0625),
			WindowEvent::KeyboardInput { event, .. } => self.hud.key_event(event),
			_ => (),
		}
	}

	pub fn on_hit(&mut self, hit: &ChangesHit) {
		self.decorations.on_hit(hit);

		if hit.killed {
			if self.smooth_world.followed_player_id().is_some_and(|id| Some(id) == hit.player_id) {
				self.sounds.play(Sfx::Kill);
			} else {
				self.sounds.play_spatial(Sfx::Kill, hit.player_pos, hit.player_vel);
			}
		} else {
			self.sounds.play_spatial(Sfx::Hit, hit.player_pos, hit.player_vel);
		}
	}

	pub fn clear_particles(&mut self) { self.decorations.clear_particles(); }

	pub fn render(&mut self, game: &mut Game, frame: &mut Frame, model: &mut Model, controller: &mut Controller, time: &TimeArgs) {
		if let (Some(px_dpos), false) = (controller.get_drag(), self.following_player()) {
			let world_dpos = self.camera.rel_pixel_to_world_coords(px_dpos);
			self.camera.move_target_dpos(world_dpos, time.dt);
		}

		/*
		 * Need to subtract dt from the time difference to make the camera move
		 * smoothly when the frame rate varies.
		 *
		 * Not too sure why this is needed, but one reason I suspect is because
		 * the calculation involves the velocity for calculating how the camera's
		 * position should exponentially decay towards the player, using the
		 * previous player's movement from last frame to this frame.
		 *
		 */
		let decay_mode = if let Some((_, player)) = self.smooth_world.followed_player() {
			self.camera.set_target(player.get_pos(time.time_diff - time.dt), player.get_vel());
			DecayMode::Player
		} else {
			self.camera.move_target_dir(controller.pcontrol.get_move_dir(), time.dt);
			DecayMode::Free
		};

		let moved_by = self.camera.update(decay_mode, time.dt);

		let matrix = self.camera.get_matrix();

		// Updates the player's cursor position with the new player position, to ensure the player is always rendered as looking towards the cursor
		if let Some(player) = self.smooth_world.current_player_mut() {
			controller.pcontrol.mouse_move_event(self.camera.pixel_to_abs_world_coords(controller.cursor_pos) - player.get_pos(time.time_diff));
			player.control(&controller.pcontrol);
		}

		// First renders the background sky, special areas and border
		self.decorations.render_sky(time.dt, &game.display, &self.camera, moved_by, frame, &mut self.params);
		self.world_renderer.render_special_areas(&game.display, frame, &mut self.params, &model.blocks, &self.camera, matrix);
		self.world_renderer.render_border(frame, &self.params, matrix);

		let forcefields = self.smooth_world.get_forcefields(time.time_diff);

		// Second renders the particles (including the death effect particles), they are decorative so they can go below everything else
		// Note that updating must come BEFORE adding new particles to avoid any new thrust particles getting a "head start" and being off-centre
		self.decorations.update_particles(time.dt);
		self.decorations.add_particles(time.dt, time.time_diff, &self.smooth_world, model.world.get_powerups(), &forcefields);
		self.decorations.render_particles(time.dt, &game.display, frame, &mut self.params, matrix);

		// Third renders everything in the world
		self.world_renderer.render(time.dt, &game.display, frame, &mut self.params, time.time_diff, &model.world, &mut self.smooth_world, &model.blocks, &forcefields, &self.camera);

		// Fourth renders the health bars, ammo status and scores, which are above everything else
		if self.render_hud {
			let names = self.hud.push_health_bars(&self.smooth_world, &self.camera, time.time_diff);
			self.hud.render(time, game, frame, &mut self.params, &model.world, &self.smooth_world, &mut model.client, &names);
		}

		self.update_sounds(time.dt);
	}

	fn update_sounds(&mut self, dt: f32) {
		self.sounds.update(self.camera.get_pos(), self.camera.get_size().min_element(), self.camera.jolted(), dt);
	}
}
