// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::fmt::Write;

use glam::Vec2;

use super::TimeArgs;
use super::view::{View, Sfx};
use super::controller::Controller;

use crate::net::{message::Message, client::{Client, ClientRes, ClientError}};
use crate::world::{World, DELTA_TIME, player::{PlayerId, update::PlayerUpdate}, changes::ClientChanges, powerup::PowerupType};
use crate::blocks::Blocks;

pub struct Model {
	id: PlayerId,
	pub(super) client: Client,
	pub(super) world: World,
	pub(super) blocks: Blocks,
}

impl Model {
	pub(super) fn new(client_res: ClientRes) -> Model {
		let (id, client, world, blocks) = client_res;
		Model { id, client, world, blocks }
	}

	pub fn get_id(&self) -> PlayerId { self.id }
	pub fn get_world(&self) -> &World { &self.world }
	pub fn get_blocks(&self) -> &Blocks { &self.blocks }

	pub(super) fn update(&mut self, view: &mut View, controller: &mut Controller, time: &TimeArgs) -> Result<(), ClientError> {
		self.client.process_messages()?;

		let init_player_pos = view.smooth_world.followed_player().map(|(_, player)| player.get_pos(time.time_diff));
		if let Some(pos) = init_player_pos {
			controller.pcontrol.mouse_move_event(view.camera.pixel_to_abs_world_coords(controller.cursor_pos) - pos);
		}
		let player_update = controller.pcontrol.get_update();

		let updates = self.client.get_world_updates();
		if !updates.is_empty() {
			let mut changes = ClientChanges::new(&self.world, self.id);
			for update in &updates {
				self.world.apply_events(&update.0.pre_events, &mut changes);
				self.world.update(&update.0.players, &self.blocks, &mut changes);
				self.world.apply_events(&update.0.post_events, &mut changes);
				self.world.validate().map_err(|err| ClientError::Limit(String::from(err)))?;
				self.client.update_timer(DELTA_TIME);
			}

			view.smooth_world.remove_old_players(&self.world);

			let ack = updates.last().unwrap().1;
			view.smooth_world.sync(&self.world, &self.blocks, &changes, ack, updates.len(), &mut view.sounds);

			if view.smooth_world.update_followed_player() {
				if let Some((_, player)) = view.smooth_world.followed_player() {
					view.camera.move_to(player.get_pos(time.time_diff));
				}
			} else if let (Some(init_pos), Some((id, player))) = (init_player_pos, view.smooth_world.followed_player()) {
				if changes.moved.contains(&id) {
					view.camera.move_by(player.get_pos(time.time_diff) - init_pos);
				}
			}

			for hit in changes.hits {
				view.on_hit(&hit);
			}

			for (sfx, pos, vel) in changes.sounds {
				view.sounds.play_spatial(sfx, pos, vel);
			}

			for (id, typ, pos) in changes.powerup_sounds {
				let sfx = Sfx::Powerup(typ);
				if typ == PowerupType::Teleportation && Some(id) == view.smooth_world.followed_player_id() {
					view.sounds.play(sfx);
				} else {
					view.sounds.play_spatial(sfx, pos, Vec2::ZERO);
				}
			}

			if changes.items_reset {
				view.clear_particles();
			}
		}

		for sfx in self.client.get_sounds() {
			view.sounds.play(sfx);
		}

		if time.steps > 0 {
			for _ in 0..time.steps {
				view.smooth_world.update(player_update, &self.world, &self.blocks, &mut view.sounds)?;
			}

			if self.world.has_player(self.id) {
				self.client.send(Message::PlayerUpdate(player_update, time.steps as u64))?;
				if self.client.is_networked() {
					self.client.send(view.smooth_world.create_bulk_update())?;
				}
			} else {
				/*
				 * NOTE: Even sending this out when spectating. This is intentional.
				 *
				 * The reason why I'm sending out player update information when
				 * spectating is so the server can acknowledge the progression of
				 * time. This allows me to reuse my current smooth world code and
				 * keep things looking smooth.
				 *
				 * I could spend some time writing code to improve smoothness in the
				 * case that the player is a spectator and possibly save bandwidth,
				 * but I can't be bothered.
				 *
				 * This information being sent out while spectating isn't correct
				 * for privacy reasons as a server admin running a modified server
				 * could capture information about what inputs the user is pressing.
				 * This isn't too serious but this information isn't needed.
				 */
				self.client.send(Message::PlayerUpdate(PlayerUpdate::default(), time.steps as u64))?;
			}
		}

		for msg in view.get_hud_mut().get_text_io().pending_messages() {
			self.client.send(Message::ClientTextInput(msg))?;
		}

		for msg in self.client.get_text_output() {
			view.get_hud_mut().get_text_io().push_history(msg);
		}

		self.client.flush()?;

		Ok(())
	}

	pub(super) fn sanity_check(&mut self) {
		if let Some(sanity_check_world) = self.client.get_sanity_check_world() {
			let mut expected_str = String::new();
			let _ = write!(&mut expected_str, "{sanity_check_world:?}");
			let mut actual_str = String::with_capacity(expected_str.len());
			let _ = write!(&mut actual_str, "{:?}", &self.world);

			if expected_str != actual_str {
				log::error!("world sync error (id = {}):\nexpected = {expected_str},\nactual = {actual_str}", self.id);
			} else {
				log::debug!("sanity check passed");
			}
		}
	}
}
