// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
pub mod model;
pub mod view;
mod controller;

use std::{time::Instant, any::Any, rc::Rc, fmt::Write};

use glium::Frame;
use winit::{event::Event, window::CursorIcon};

use model::Model;
use view::View;
use controller::Controller;

use crate::game::{Game, config::action::{Action, ActionEvent, ActionState}, state::{GameState, Status}};
use crate::net::{client::{ClientRes, ClientError}, message_stream::MsError};

struct TimeArgs {
	dt: f32,
	steps: u32,
	time_diff: f32,
}

pub struct Playing<'a> {
	model: Option<Model>,
	view: View<'a>,
	controller: Controller,

	leave: bool,
	prev_cursor: Option<CursorIcon>,
}

impl<'a> Playing<'a> {
	pub fn new(game: &mut Game) -> Playing<'a> {
		let view = View::new(Rc::clone(&game.config), game);
		let controller = Controller::new(Instant::now());

		Playing { model: None, view, controller, leave: false, prev_cursor: None }
	}

	fn action_text_input_closed(&mut self, action: ActionEvent) {
		if matches!(action, ActionEvent::Action(Action::PlayingLeave, ActionState::Pressed)) {
			self.leave = true;
		}

		if matches!(action, ActionEvent::Action(Action::PrintPos, ActionState::Pressed)) {
			let world_pos = self.view.camera.pixel_to_abs_world_coords(self.controller.cursor_pos);

			let mut msg = String::from("Cursor: World = ");
			let _ = write!(msg, "{world_pos}");

			if let Some(model) = &self.model {
				let (block_pos, in_bounds) = model.blocks.world_to_block_pos(world_pos);
				let _ = write!(msg, ", Block = {block_pos}");
				if !in_bounds {
					let _ = write!(msg, " (out of bounds)");
				}
			}

			log::info!("{msg}");
		}

		self.view.action(action);
		self.controller.action(action);
	}
}

impl GameState for Playing<'_> {
	fn enable(&mut self, game: &mut Game) {
		game.gui.do_extra_pass();
		self.view.sounds.set_muted(false);
		self.prev_cursor = None;
	}

	fn push(&mut self, game: &mut Game, msg: Option<Box<dyn Any>>) {
		let model = Model::new(*msg.unwrap().downcast::<ClientRes>().unwrap());
		self.view.reset(game, &model);
		self.controller.reset(Instant::now());
		self.model = Some(model);
	}

	fn disable(&mut self, game: &mut Game) {
		self.model = None;
		self.view.sounds.set_muted(true);
		game.window.set_cursor(CursorIcon::Default);
	}

	fn action(&mut self, action: ActionEvent) {
		let prev_open = self.view.get_hud().text_input_open();
		self.view.text_input_action(action);

		if self.view.get_hud().text_input_open() {
			if !prev_open {
				self.action_text_input_closed(ActionEvent::AllReleased);
			}
		} else {
			self.action_text_input_closed(action);
		}
	}

	fn event(&mut self, game: &mut Game, event: &Event<()>) {
		match event {
			Event::WindowEvent { event, .. } => {
				self.view.window_event(event);
				self.controller.window_event(event);
			},
			Event::AboutToWait => game.window.request_redraw(),
			_ => (),
		}
	}

	fn loop_iter(&mut self, game: &mut Game, frame: &mut Frame) -> Status {
		let time = self.controller.get_time();

		let model = self.model.as_mut().unwrap();

		if let Err(err) = model.update(&mut self.view, &mut self.controller, &time) {
			match err {
				ClientError::Io(MsError::Disconnected(Some(reason))) => log::warn!("server disconnected: {reason}"),
				ClientError::Io(MsError::Disconnected(None)) => log::warn!("server disconnected"),
				ClientError::Io(MsError::Other(msg)) => log::warn!("server error and disconnection: {msg}"),
				ClientError::Protocol(msg) => log::warn!("disconnecting after server violated protocol: {msg}"),
				ClientError::Limit(msg) => log::warn!("disconnecting after client limit violated: {msg}"),
			}

			self.leave = true;
		} else {
			model.sanity_check();
		}

		let cursor =
			if self.view.following_player() { CursorIcon::Crosshair }
			else if self.controller.is_dragging() { CursorIcon::Grabbing }
			else { CursorIcon::Grab };

		if Some(cursor) != self.prev_cursor {
			game.window.set_cursor(cursor);
			self.prev_cursor = Some(cursor);
		}

		self.view.render(game, frame, model, &mut self.controller, &time);

		if self.leave {
			self.leave = false;
			Status::PopState
		} else { Status::Ok }
	}
}
