// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::collections::{BTreeMap, btree_map::Entry};

use crate::world::{World, player::{PlayerId, PlayerStyle}, team::TeamId};

#[derive(Default)]
pub struct Joined(BTreeMap<PlayerId, JoinState>);

#[derive(Clone)]
pub struct JoinInfo {
	pub style: PlayerStyle,
	pub team: Option<TeamId>,
}

impl JoinInfo {
	pub fn update_team(&mut self, team_id: TeamId, world: &World) {
		let Some(team) = world.get_team(team_id) else { return; };
		self.team = Some(team_id);
		self.style.colour = team.get_colour();
	}
}

enum JoinState {
	NewClientPlaying(JoinInfo), // A client has joined as a player
	OldClientPlaying(JoinInfo), // A previously spectating client has started playing
	NewClientSpectating, // A client has joined as a spectator
}

impl Joined {
	pub(super) fn add_client_spectating(&mut self, id: PlayerId) {
		let prev = self.0.insert(id, JoinState::NewClientSpectating);
		debug_assert!(prev.is_none());
	}

	pub(super) fn add_client_playing(&mut self, id: PlayerId, info: JoinInfo) {
		let prev = self.0.insert(id, JoinState::NewClientPlaying(info));
		debug_assert!(prev.is_none());
	}

	pub(super) fn add_player(&mut self, id: PlayerId, info: JoinInfo) {
		match self.0.entry(id) {
			Entry::Occupied(mut e) => {
				match e.get() {
					// Don't want to forget that this is a new client, or the initial messages won't be sent out
					JoinState::NewClientPlaying(..) | JoinState::OldClientPlaying(..) => (),
					JoinState::NewClientSpectating => _ = e.insert(JoinState::NewClientPlaying(info)),
				}
			},
			Entry::Vacant(e) => _ = e.insert(JoinState::OldClientPlaying(info)),
		}
	}

	pub(super) fn is_new_client(&self, id: PlayerId) -> bool {
		match self.0.get(&id) {
			Some(JoinState::NewClientPlaying(..) | JoinState::NewClientSpectating) => true,
			Some(JoinState::OldClientPlaying(..)) | None => false,
		}
	}

	pub fn players(&self) -> impl Iterator<Item = (PlayerId, JoinInfo)> + '_ {
		self.0
			.iter()
			.filter_map(|(&id, state)| match state {
				JoinState::NewClientPlaying(info) | JoinState::OldClientPlaying(info) => Some((id, info.clone())),
				JoinState::NewClientSpectating => None,
			})
	}
}
