// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::{collections::{BTreeMap, BTreeSet, btree_map::Entry}, net::SocketAddr, num::{NonZeroU16, NonZeroUsize}, sync::{Arc, atomic::{AtomicUsize, Ordering}}, fmt::Write};

use tokio::sync::mpsc::{self, Sender, error::TrySendError};
use nonempty::NonEmpty;

use super::repl::command::RouteCommand;

use crate::net::{server::{self, Server, ServerMessage}, connection_listener::networked::NetworkedConnectionListener, lan_discovery::{LanServerInfo, PROTOCOL_VERSION}};
use crate::net::message_stream::quic_server::QuicServerMsIncoming;
use crate::net::serp::request::{PlayRequest, PLAY_VERSION, request_types};
use crate::net::text_io::{TextIoString, TextIoStr, MAX_LEN};
use crate::utils::INDENTATION;

pub struct GameServer {
	stream_sender: Sender<(QuicServerMsIncoming, PlayRequest)>,
	message_sender: Sender<ServerMessage>,
	path: Arc<str>,
	game_ids: BTreeSet<Arc<str>>,
	player_count: Arc<AtomicUsize>,
	max_players: Option<NonZeroUsize>,
}

impl GameServer {
	pub async fn build(addr: SocketAddr, entry: &RouterEntry, sanity_checks: bool, use_lan_discovery: bool) -> Result<GameServer, String> {
		let config = server::load_config_async(entry.path.as_ref()).await.map_err(|err| format!("failed loading config: {err}"))?;
		let max_players = config.max_players;

		let info = LanServerInfo {
			version: PROTOCOL_VERSION,
			request_type: request_types::PLAY,
			request_version: PLAY_VERSION,
			port: addr.port(),
			cur_players: 0,
			max_players: config.max_players.and_then(|max| u16::try_from(max.get()).ok()).and_then(NonZeroU16::new),
			game_id: Box::from(""),
			name: config.lan_discovery.name.clone().into_boxed_str(),
			desc: config.lan_discovery.desc.clone().into_boxed_str(),
		};

		let (stream_sender, stream_receiver) = mpsc::channel(256);

		let connection_listener = NetworkedConnectionListener::new(stream_receiver);
		let server = Server::build(Box::new(connection_listener), config, sanity_checks).map_err(|err| format!("failed creating server: {err}"))?;

		let lan_info = match info.validate() {
			_ if !use_lan_discovery => None,
			Ok(info) => Some(info),
			Err(err) => {
				log::warn!("invalid LAN discovery config for server at \"{}\": {err}", entry.path);
				None
			},
		};

		let (message_sender, message_receiver) = mpsc::channel(256);

		let player_count = Arc::new(AtomicUsize::new(0));
		tokio::spawn(server.run(message_receiver, lan_info, Arc::clone(&player_count)));

		Ok(GameServer {
			stream_sender,
			message_sender,
			path: Arc::clone(&entry.path),
			game_ids: entry.game_ids.iter().map(Arc::clone).collect(),
			player_count,
			max_players,
		})
	}

	fn send_admin_message(&self, msg: TextIoString) -> Result<(), String> {
		match self.message_sender.try_send(ServerMessage::AdminMessage(msg)) {
			/*
			 * If the server closed (I don't think it's possible for this to happen
			 * while in the router, don't bother reporting the error to the user as
			 * it's not important. This is because an error is worth reporting if
			 * users who should see a message don't get the message, which isn't
			 * the case here as the server's closed.
			 */
			Ok(()) | Err(TrySendError::Closed(_)) => Ok(()),
			Err(TrySendError::Full(_)) => Err(String::from("failed sending admin message to at least one server: sender is full")),
		}
	}
}

pub struct RouterEntry {
	path: Arc<str>,
	game_ids: NonEmpty<Arc<str>>,
}

impl RouterEntry {
	/**
	 * Parses a colon-separated string consisting of "id:id:...:id:path" and
	 * splits up the game ids and paths. If there are no colons, then the server
	 * is treated as the default game server, so it's given the id "default".
	 *
	 * A game server can have the "default" id and multiple others. Those extra
	 * ids don't have any effect on their own, but they can prevent other servers
	 * from being created with those ids, and if the default id is removed, those
	 * ids can still be used.
	 */
	pub fn build(game_server: &str) -> Result<RouterEntry, String> {
		let mut parts: Vec<Arc<str>> = game_server.split(':').map(Arc::from).collect();
		let path = parts.pop().expect("split should return at least one part"); // Shouldn't ever panic

		let game_ids = NonEmpty::from_vec(parts).unwrap_or_else(|| NonEmpty::new(Arc::from("default")));
		let mut set = BTreeSet::new();

		// Don't want any duplicates as they might cause issues later on
		for game_id in &game_ids {
			if !set.insert(game_id.as_ref()) {
				return Err(format!("duplicate game id \"{game_id}\""));
			}
		}

		Ok(RouterEntry { path, game_ids })
	}
}

#[derive(Default)]
pub struct Router { // Routes game ids to servers
	map: BTreeMap<Arc<str>, usize>,
	servers: BTreeMap<usize, GameServer>, // Not using a Vec<GameServer> because I want the index to be constant when servers are removed
	next_index: usize,
}

impl Router {
	/**
	 * Adds a new game server to the router. Keeps it unmodified and in a
	 * consistent state on error.
	 */
	pub fn add_game_server(&mut self, entry: RouterEntry, server: GameServer) -> Result<(), String> {
		// First ensures the game id isn't previously used
		for game_id in &entry.game_ids {
			if self.map.contains_key(game_id) {
				// Returns keeping things unmodified
				return Err(format!("game id \"{game_id}\" used multiple times"));
			}
		}

		while self.servers.contains_key(&self.next_index) {
			self.next_index = self.next_index.wrapping_add(1);
		}

		self.servers.insert(self.next_index, server);
		for game_id in entry.game_ids { self.map.insert(game_id, self.next_index); }

		Ok(())
	}

	pub(super) fn remove_game_server(&mut self, game_id: &str) -> Result<(), String> {
		if let Some(index) = self.map.get(game_id) {
			let server = self.servers.remove(index).unwrap();
			for game_id in &server.game_ids {
				let found = self.map.remove(game_id.as_ref()).is_some();
				debug_assert!(found);
			}
			Ok(())
		} else {
			Err(format!("game id \"{game_id}\" not found"))
		}
	}

	pub(super) fn get_router_entry(&self, game_id: &str) -> Option<RouterEntry> {
		let server = &self.servers[self.map.get(game_id)?];
		Some(RouterEntry {
			path: Arc::clone(&server.path),
			game_ids: NonEmpty::from_vec(server.game_ids.iter().map(Arc::clone).collect()).unwrap(),
		})
	}

	pub(super) fn reload_game_server(&mut self, server: GameServer) -> Result<(), String> {
		/*
		 * Checking for errors and not unwrapping in this method as the read lock
		 * is released after first getting the router entry, so it's possible that
		 * somewhere else in the code modifies the server config. I don't want to
		 * hold that lock while loading a server, as that could take a while
		 * (I/O).
		 *
		 * Right now in the current configuration I don't think this can change as
		 * the router is only modified through the REPL, but that could change.
		 */
		let any_id = server.game_ids.first().unwrap().as_ref();
		let index = self.map.get(any_id).ok_or_else(|| String::from("cannot find old server (possible race condition?)"))?;
		let old_server = &self.servers[index];

		if old_server.game_ids != server.game_ids { return Err(String::from("game ids of old and new server don't match (possible race condition?)")); }
		if old_server.path != server.path { return Err(String::from("paths of old and new server don't match (possible race condition?)")); }

		self.servers.insert(*index, server);

		Ok(())
	}

	pub(super) fn route(&self, game_id: &str) -> Option<(usize, Sender<(QuicServerMsIncoming, PlayRequest)>)> {
		self.map.get(game_id).or_else(|| self.map.get("default")).map(|index| (*index, self.servers[index].stream_sender.clone()))
	}

	pub(super) fn change_route(&mut self, cmd: RouteCommand) -> Result<(), String> {
		match cmd {
			RouteCommand::Add { id, new_id } => {
				let Some(&index) = self.map.get(id.as_ref()) else {
					return Err(format!("failed adding route: game id \"{id}\" does not exist"));
				};

				if self.map.contains_key(new_id.as_ref()) {
					return Err(format!("failed adding route: new game id \"{new_id}\" already exists"));
				}

				let server = self.servers.get_mut(&index).unwrap();

				let new_id = Arc::from(new_id);
				let not_found = server.game_ids.insert(Arc::clone(&new_id));
				debug_assert!(not_found);

				let not_found = self.map.insert(new_id, index).is_none();
				debug_assert!(not_found);
				Ok(())
			},
			RouteCommand::Remove(id) => {
				let map_entry = match self.map.entry(Arc::from(id)) {
					Entry::Occupied(e) => e,
					Entry::Vacant(e) => return Err(format!("failed removing route: game id \"{}\" does not exist", e.key())),
				};
				let (id, index) = (map_entry.key(), *map_entry.get());

				let server = self.servers.get_mut(&index).unwrap();
				match server.game_ids.len() {
					0 => unreachable!(),
					1 => return Err(format!("failed removing route: game server with id \"{id}\" cannot have route removed, as doing so would remove the server (to remove the server, run `remove {id}`)")),
					_ => (),
				}

				let found = server.game_ids.remove(id);
				debug_assert!(found);

				map_entry.remove();
				Ok(())
			},
		}
	}

	pub(super) fn send_admin_message(&self, msg: &str, game_ids: Option<NonEmpty<Box<str>>>) -> Result<(), String> {
		let mut ret = Ok(()); // If there's an error, don't stop sending to all other servers

		let Some(msg) = TextIoStr::new(msg) else {
			return Err(format!("admin message too long, expected at most {MAX_LEN} bytes, got {}", msg.len()));
		};

		if let Some(ids) = game_ids { // Sends the message to a particular server
			// Using a set to avoid sending the same message multiple times to a server
			let mut indices = BTreeSet::new();
			for id in ids {
				if let Some(index) = self.map.get(id.as_ref()) {
					indices.insert(index);
				} else {
					return Err(format!("failed sending admin message: game id \"{id}\" does not exist"));
				}
			}

			for index in indices {
				if let Err(err) = self.servers[index].send_admin_message(msg.owned()) {
					ret = Err(err);
				}
			}
		} else { // Sends the message to all servers
			for server in self.servers.values() {
				if let Err(err) = server.send_admin_message(msg.owned()) {
					ret = Err(err);
				}
			}
		}

		ret
	}

	// Returning a string rather than printing as I don't want to perform I/O while a lock's being held
	pub(super) fn get_status(&self, game_id: Option<&str>) -> String {
		let mut status = String::new();
		if let Some(id) = game_id { // Get the status for just that id
			if let Some(&index) = self.map.get(id) {
				let _ = writeln!(&mut status, "Showing status for game server with id \"{id}\":");
				self.write_status_for_index(&mut status, index);
			} else {
				let _ = writeln!(&mut status, "No game server with id \"{id}\" exists.");
			}
		} else if !self.servers.is_empty() {
			let _ = writeln!(&mut status, "Showing status for all game servers:");
			for (&index, server) in &self.servers {
				Router::write_status_for_index_and_server(&mut status, index, server);
				let _ = writeln!(&mut status);
			}
		} else {
			let _ = writeln!(&mut status, "No game servers are running.");
		}

		status
	}

	fn write_status_for_index(&self, status: &mut String, index: usize) {
		Router::write_status_for_index_and_server(status, index, &self.servers[&index]);
	}

	fn write_status_for_index_and_server(status: &mut String, index: usize, server: &GameServer) {
		let _ = write!(status, "{index}: {:?}\n{}path = {}\n{INDENTATION}client count = {}", server.game_ids, INDENTATION, server.path, server.player_count.load(Ordering::Relaxed));
		if let Some(max) = server.max_players {
			let _ = write!(status, "/{max}");
		}
		let _ = writeln!(status);
	}

	/**
	 * Returns a Vec of senders used to send `ServerMessage`s to the game server.
	 * Also includes an associated game id that's up-to-date which is used for
	 * LAN discovery.
	 */
	pub(super) fn get_server_message_send_info(&self) -> Vec<(Sender<ServerMessage>, Arc<str>)> {
		self.servers
			.values()
			.map(|server| (server.message_sender.clone(), server.game_ids.first().map(Arc::clone).unwrap_or_default() /* Shouldn't fail but I don't want to panic */))
			.collect()
	}
}
