// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use strum_macros::EnumIter;
use const_format::formatcp;
use nonempty::NonEmpty;

use super::super::{CertInfo, router::RouterEntry};

use crate::utils::command::{Args, Command as GenericCommand, CommandVariant as GenericCommandVariant, HELP_FOR_HELP};

pub enum Command {
	Help(Option<CommandVariant>),
	Exit,
	Clear,
	Add(RouterEntry),
	Remove(Box<str>), // Removes by id
	Reload(Box<str>),
	Route(RouteCommand),
	Status(Option<Box<str>>),
	Cert(CertInfo),
	Info,
	Message(Box<str>, Option<NonEmpty<Box<str>>>),
}

impl GenericCommand for Command {}

#[derive(Clone, Copy, EnumIter)]
pub enum CommandVariant { Help, Exit, Clear, Add, Remove, Reload, Route, Status, Cert, Info, Message }

impl GenericCommandVariant for CommandVariant {
	type Command = Command;

	fn try_from(cmd: &str) -> Option<CommandVariant> {
		match cmd {
			"help" => Some(CommandVariant::Help),
			"exit" => Some(CommandVariant::Exit),
			"clear" => Some(CommandVariant::Clear),
			"add" => Some(CommandVariant::Add),
			"remove" => Some(CommandVariant::Remove),
			"reload" => Some(CommandVariant::Reload),
			"route" => Some(CommandVariant::Route),
			"status" => Some(CommandVariant::Status),
			"cert" => Some(CommandVariant::Cert),
			"info" => Some(CommandVariant::Info),
			"message" => Some(CommandVariant::Message),
			_ => None,
		}
	}

	fn help(self) -> &'static str {
		use crate::utils::INDENTATION as I;

		match self {
			CommandVariant::Help => formatcp!("{HELP_FOR_HELP}\n"),
			CommandVariant::Exit => "exit - exits the server\n",
			CommandVariant::Clear => "clear - clears the screen (you can also do this with Ctrl+L)\n",
			CommandVariant::Add => formatcp!("add - adds a new game server\n{I}add <ID>:<ID>:...:<ID>:<PATH>: Loads a game server from PATH with ID as each id. If there are no ids, the id \"default\" is used, which routes everything else to it.\n"),
			CommandVariant::Remove => formatcp!("remove - removes an existing game server\n{I}remove <ID>: Removes the game server with id of ID.\n"),
			CommandVariant::Reload => formatcp!("reload - reloads the configuration of an existing game server\n{I}reload <ID>: Creates a new game server using the configuration file from the server with ID. Then replaces that server with the newly loaded server on success, preserving all routing information.\n"),
			CommandVariant::Route => formatcp!("route - change routing information\n{I}route add <ID> <NEW_ID>: Routes NEW_ID to the game server with id of ID. ID must correspond with an existing game server and NEW_ID must not already be used.\n{I}route remove <ID>: Removes the routing rule from ID to its corresponding game server. ID must exist and the server with that id must have another id so it's still reachable.\n"),
			CommandVariant::Status => formatcp!("status - prints information about running game servers\n{I}status: Prints information about all servers.\n{I}status <ID>: Prints information about a server with the id of ID.\n"),
			CommandVariant::Cert => formatcp!("cert - updates certificate information\n{I}cert <CERT_PATH> <KEY_PATH>: Updates the TLS certificate used by the server, taking paths to files in the PEM format.\n"),
			CommandVariant::Info => "info - prints the publicly available build information about the server\n",
			CommandVariant::Message => formatcp!("message - sends a message to users in the game\n{I}message <MESSAGE>: Sends a message to all game servers.\n{I}message <IDS> <MESSAGE>Sends a message to the servers with game ids as provided in the previous arguments.\n"),
		}
	}

	fn build(self, args: Args) -> Result<Command, String> {
		Ok(match self {
			CommandVariant::Help => Command::Help(args.help_command::<CommandVariant>()?),
			CommandVariant::Exit => args.require_none(Command::Exit)?,
			CommandVariant::Clear => args.require_none(Command::Clear)?,
			CommandVariant::Add => {
				let game = &args.require_n_args::<1>()?[0];
				Command::Add(RouterEntry::build(game)?)
			},
			CommandVariant::Remove => Command::Remove(args.require_n_args::<1>()?[0].clone().into_boxed_str()),
			CommandVariant::Reload => Command::Reload(args.require_n_args::<1>()?[0].clone().into_boxed_str()),
			CommandVariant::Route => {
				let subcmd = args.get_arg(0)?;
				match subcmd {
					"add" => {
						let [_, id, new_id] = args.require_n_args_subcmd::<3>(subcmd)?;
						Command::Route(RouteCommand::Add { id: id.clone().into_boxed_str(), new_id: new_id.clone().into_boxed_str() })
					},
					"remove" => {
						let [_, id] = args.require_n_args_subcmd::<2>(subcmd)?;
						Command::Route(RouteCommand::Remove(id.clone().into_boxed_str()))
					},
					_ => return Err(format!("unknown subcommand of \"{subcmd}\"")),
				}
			},
			CommandVariant::Status => {
				args.check_at_most_n_args(1)?;
				Command::Status(args.first().map(|id| id.clone().into_boxed_str()))
			},
			CommandVariant::Cert => {
				let [cert_path, key_path] = args.require_n_args::<2>()?;
				Command::Cert(CertInfo { cert_path: cert_path.clone().into_boxed_str(), key_path: key_path.clone().into_boxed_str() })
			},
			CommandVariant::Info => args.require_none(Command::Info)?,
			CommandVariant::Message => {
				let Some(message) = args.last() else {
					return Err(String::from("expected at least one argument, got none"));
				};

				let ids = args.iter().take(args.len() - 1).cloned().map(String::into_boxed_str).collect();
				Command::Message(message.clone().into_boxed_str(), NonEmpty::from_vec(ids))
			},
		})
	}
}

pub enum RouteCommand {
	Add {
		id: Box<str>,
		new_id: Box<str>,
	},
	Remove(Box<str>),
}

#[cfg(test)]
mod tests {
	use strum::IntoEnumIterator;

	use super::*;

	#[test]
	fn all_help_text_ends_in_newline() {
		for cmd in CommandVariant::iter() {
			assert!(cmd.help().ends_with('\n'));
		}
	}
}
