// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
pub mod command;

use std::{net::SocketAddr, sync::Arc, io::{self, Write}};

use tokio::runtime::Handle;
use quinn::ServerConfig;
use linefeed::{Interface, ReadResult};
use strum::IntoEnumIterator;

use command::{Command, CommandVariant};

use super::{ServerEndpoint, super::{CertInfo, SHUTTING_DOWN_MSG, SHUTTING_DOWN_STATUS}, router::{GameServer, RouterEntry}};

use crate::ServerArgs;
use crate::utils::command::{Command as _, CommandVariant as _};

pub struct Repl {
	endpoint: Arc<ServerEndpoint>,
	handle: Handle,
	args: ServerArgs,
	addr: SocketAddr,
}

impl Repl {
	pub fn new(endpoint: Arc<ServerEndpoint>, args: ServerArgs, addr: SocketAddr) -> Repl {
		Repl { endpoint, handle: Handle::current(), args, addr }
	}

	pub fn run(self) {
		let interface = Interface::new("spaceships").unwrap();
		interface.set_prompt("> ").unwrap();

		loop {
			match interface.read_line() {
				Ok(ReadResult::Input(line)) => {
					match Command::build::<CommandVariant>(&line) {
						Ok(Some(Command::Help(None))) => {
							println!("Showing help of all commands:");
							for cmd in CommandVariant::iter() {
								print!("{}", cmd.help());
							}
						},
						Ok(Some(Command::Help(Some(cmd)))) => print!("{}", cmd.help()),
						Ok(Some(Command::Exit)) => break,
						Ok(Some(Command::Clear)) => {
							/*
							 * Would like to use the `clear_screen` method that this
							 * crate has, but I can't seem to access the type that
							 * allows it.
							 */
							print!("\x1b[H\x1b[2J\x1b[3J"); // Output of `clear | xxd`, might not work on all platforms
							let _ = io::stdout().flush();
						},
						Ok(Some(Command::Add(entry))) => {
							if let Err(err) = self.add_game_server(entry) {
								println!("error: add: {err}");
							}
						},
						Ok(Some(Command::Remove(game_id))) => {
							let res = self.endpoint.router.write().unwrap().remove_game_server(game_id.as_ref()); // Ensures that the lock is dropped as soon as possible
							if let Err(err) = res {
								println!("error: remove: cannot remove game server: {err}");
							}
						},
						Ok(Some(Command::Reload(game_id))) => {
							if let Err(err) = self.reload_game_server(game_id) {
								println!("error: reload: {err}");
							}
						},
						Ok(Some(Command::Route(cmd))) => {
							let res = self.endpoint.router.write().unwrap().change_route(cmd);
							if let Err(err) = res {
								println!("error: route: {err}");
							}
						},
						Ok(Some(Command::Status(id))) => {
							let status = self.endpoint.router.read().unwrap().get_status(id.as_ref().map(Box::as_ref));
							print!("{status}");
						},
						Ok(Some(Command::Cert(info))) => {
							if let Err(err) = self.update_cert_info(info) {
								println!("error: cert: {err}");
							} else {
								println!("Successfully updated certificate information!");
							}
						},
						Ok(Some(Command::Info)) => self.endpoint.info.print(),
						Ok(Some(Command::Message(msg, game_ids))) => {
							let res = self.endpoint.router.read().unwrap().send_admin_message(msg.as_ref(), game_ids);
							if let Err(err) = res {
								println!("error: message: {err}");
							}
						},
						Ok(None) => (),
						Err(err) => println!("error: {err}"), // Not using stderr as that's for logs, and if redirecting logs to a file, those will be ignored
					}

					interface.add_history(line);
				},
				Ok(ReadResult::Signal(_)) => (),
				Ok(ReadResult::Eof) => break,
				Err(err) => {
					println!("error reading line, exiting early: {err}");
					break;
				},
			}
		}

		self.endpoint.endpoint.close(SHUTTING_DOWN_STATUS, SHUTTING_DOWN_MSG);
	}

	fn create_game_server(&self, entry: &RouterEntry) -> Result<GameServer, String> {
		self.handle.block_on(GameServer::build(self.addr, entry, self.args.sanity_checks, self.endpoint.lan_discovery.is_some())).map_err(|err| format!("failed creating game server: {err}"))
	}

	fn add_game_server(&self, entry: RouterEntry) -> Result<(), String> {
		let server = self.create_game_server(&entry)?;
		self.endpoint.router.write().unwrap().add_game_server(entry, server).map_err(|err| format!("failed adding game server: {err}"))
	}

	fn reload_game_server(&self, game_id: Box<str>) -> Result<(), String> {
		let entry = self.endpoint.router.read().unwrap().get_router_entry(game_id.as_ref()).ok_or_else(|| format!("game id \"{game_id}\" does not exist"))?;
		let server = self.create_game_server(&entry)?;
		self.endpoint.router.write().unwrap().reload_game_server(server)
	}

	fn update_cert_info(&self, info: CertInfo) -> Result<(), String> {
		let (certs, key) = self.handle.block_on(ServerEndpoint::load_cert_info(info)).map_err(|err| format!("failed loading certificate info: {err}"))?;
		let config = ServerConfig::with_single_cert(certs, key).map_err(|err| format!("cannot create server config: {err}"))?;
		self.endpoint.endpoint.set_server_config(Some(config));
		Ok(())
	}
}
