// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::sync::Arc;

use tokio::sync::oneshot;

use super::ServerEndpoint;

use crate::net::{server::ServerMessage, lan_discovery::{server::LanDiscoveryServerRecvBuf, security::SecurityPolicy}, utils::SocketAddrToCanonical};

impl ServerEndpoint {
	pub(super) fn run_lan_discovery(self: &Arc<Self>) {
		if let Some(lan_discovery) = &self.lan_discovery {
			let lan_discovery = lan_discovery.clone();
			let endpoint = Arc::clone(self);

			tokio::spawn(async move {
				let mut policy = SecurityPolicy::from_config(endpoint.lan_discovery_config);
				let mut recv_buf = LanDiscoveryServerRecvBuf::new();
				loop {
					match lan_discovery.receive(&mut recv_buf, &mut policy).await {
						Ok(addr) => {
							log::info!("Received LAN discovery message from {}", addr.to_canonical());

							let senders = endpoint.router.read().unwrap().get_server_message_send_info();
							for (sender, game_id) in senders {
								let lan_discovery = lan_discovery.clone();

								tokio::spawn(async move {
									let (oneshot_sender, oneshot_receiver) = oneshot::channel();
									if sender.send(ServerMessage::LanDiscovery(oneshot_sender)).await.is_err() {
										return;
									}

									let Ok(mut info) = oneshot_receiver.await else { return; };
									info.game_id = Box::from(game_id.as_ref());

									if let Err(err) = lan_discovery.respond(info, addr).await {
										log::warn!("{err}");
									}
								});
							}
						},
						Err(err) => {
							log::warn!("{err}");
							return;
						},
					}
				}
			});
		}
	}
}
