// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
mod lan_discovery;
pub(super) mod router;
mod repl;

pub use repl::Repl;

use std::{net::SocketAddr, sync::{Arc, RwLock}};

use tokio::{io::{BufWriter, AsyncWriteExt}, fs};
use quinn::{Endpoint, Incoming, SendStream, VarInt, ServerConfig, EndpointConfig};
use rustls::pki_types::{CertificateDer, PrivateKeyDer, PrivatePkcs8KeyDer};
use bincode::Options;

use router::Router;

use super::CertInfo;

use crate::LanDiscoveryConfig;
use crate::net::{
	serp::{Request, Response, Vu30, vu30::AsyncWriteVu30}, info::Info, message_stream::{quic_common, quic_server::QuicServerMsIncoming},
	lan_discovery::server::LanDiscoveryServer, utils::{WaitIdleTimeout, SocketAddrToCanonical, udp}
};

pub struct ServerEndpoint {
	endpoint: Endpoint,
	lan_discovery: Option<LanDiscoveryServer>,
	lan_discovery_config: LanDiscoveryConfig,
	router: RwLock<Router>,
	info: Info,
}

impl ServerEndpoint {
	pub async fn build(addr: SocketAddr, cert_info: Option<CertInfo>, router: Router, lan_discovery: Option<LanDiscoveryServer>, lan_discovery_config: LanDiscoveryConfig) -> Result<Arc<ServerEndpoint>, String> {
		// Loads certificate information from the provided paths, otherwise using a self-signed certificate
		let (certs, key) = if let Some(info) = cert_info {
			ServerEndpoint::load_cert_info(info).await.map_err(|err| format!("failed loading certificate info: {err}"))?
		} else {
			let cert = rcgen::generate_simple_self_signed(vec![String::from("localhost")]).map_err(|err| format!("cannot generate self-signed certificate: {err}"))?;
			let key = PrivatePkcs8KeyDer::from(cert.key_pair.serialize_der());
			(vec![cert.cert.into()], key.into())
		};

		let config = ServerConfig::with_single_cert(certs, key).map_err(|err| format!("cannot create server config: {err}"))?;
		let socket = udp::new(addr).map_err(|err| format!("failed binding socket on {addr}: {err}"))?;
		let runtime = quinn::default_runtime().ok_or_else(|| String::from("failed getting tokio runtime"))?;
		let endpoint = Endpoint::new(EndpointConfig::default(), Some(config), socket, runtime).map_err(|err| format!("cannot create QUIC server endpoint on {addr}: {err}"))?;

		Ok(Arc::new(ServerEndpoint { endpoint, lan_discovery, lan_discovery_config, router: RwLock::new(router), info: Info::new() }))
	}

	// Copied and modified from https://github.com/quinn-rs/quinn/blob/main/quinn/examples/server.rs
	async fn load_cert_info(info: CertInfo) -> Result<(Vec<CertificateDer<'static>>, PrivateKeyDer<'static>), String> {
		let cert_data = fs::read(info.cert_path.as_ref()).await.map_err(|err| format!("failed loading certificate chain at \"{}\": {err}", info.cert_path))?;
		let certs = rustls_pemfile::certs(&mut &*cert_data)
			.collect::<Result<_, _>>()
			.map_err(|err| format!("failed parsing certificates, make sure they're in PEM format: {err}"))?;

		let key_data = fs::read(info.key_path.as_ref()).await.map_err(|err| format!("failed loading private key at \"{}\": {err}", info.key_path))?;
		let key = match rustls_pemfile::private_key(&mut &*key_data) {
			Ok(Some(key)) => key,
			Ok(None) => return Err(String::from("no private key found")),
			Err(err) => return Err(format!("failed parsing private key, make sure it's in the PEM format: {err}")),
		};

		Ok((certs, key))
	}

	pub async fn run(self: Arc<Self>) {
		self.run_lan_discovery();

		while let Some(incoming) = self.endpoint.accept().await {
			let endpoint = Arc::clone(&self);
			tokio::spawn(endpoint.handle_incoming(incoming));
		}

		self.endpoint.wait_idle_timeout().await;
	}

	async fn handle_incoming(self: Arc<Self>, incoming: Incoming) {
		let addr = incoming.remote_address().to_canonical();
		match incoming.await {
			Ok(connection) => {
				log::info!("connection attempt, addr = {addr}");

				// Doesn't seem to do anything, but maybe that's because my testing had the streams created immediately
				connection.set_max_concurrent_bi_streams(VarInt::from_u32(1));
				connection.set_max_concurrent_uni_streams(VarInt::from_u32(0));

				match Request::receive(&connection).await {
					Ok((Request::Play(request), subrequest_index, mut sender, receiver)) => {
						let router_res = self.router.read().unwrap().route(&request.game_id);
						if let Some((index, stream_sender)) = router_res {
							log::info!("play request received, addr = {addr}, game id = {}, routing to server {index}", request.game_id);

							let Ok(stream) = QuicServerMsIncoming::build(connection, sender, receiver, subrequest_index).await.inspect_err(|err|
								log::info!("creating message stream failed, addr = {addr}, game id = {}, server index = {index}: {err}", request.game_id)
							) else { return; };

							let _ = stream_sender.send((stream, request)).await;
						} else {
							log::info!("connection failed, no server matches game id, addr = {addr}, game id = {}", request.game_id);
							if let Err(err) = Response::Understood(subrequest_index, String::from("no server matches game id")).reply(&mut sender).await {
								log::info!("failed sending understood response, addr = {addr}: {err}");
							}
						}
					},
					Ok((Request::Info, index, sender, _receiver)) => {
						log::info!("info request received, addr = {addr}");
						if let Err(err) = ServerEndpoint::respond_info_request(sender, index, &self.info).await {
							log::info!("sending server info failed, addr = {addr}: {err}");
						}
					},
					Err(Ok((status, mut sender))) => {
						log::info!("connection failed with status, addr = {addr}");
						if let Err(err) = status.reply(&mut sender).await {
							log::info!("sending server info failed, addr = {addr}: {err}");
						}
					},
					Err(Err(err)) => log::info!("connection failed, addr = {addr}: {err}"),
				}
			},
			Err(err) => log::info!("connection attempt failed, addr = {addr}: {err}"),
		}
	}

	async fn respond_info_request(mut sender: BufWriter<SendStream>, index: Vu30, info: &Info) -> Result<(), String> {
		Response::Success(index).reply(&mut sender).await.map_err(|err| format!("failed sending success response: {err}"))?;
		let info_data = quic_common::bincode_options().serialize(info).map_err(|err| format!("failed serialising info: {err}"))?;
		sender.write_bytes(&info_data).await.map_err(|err| format!("failed sending info: {err}"))?;
		sender.flush().await.map_err(|err| format!("failed flushing: {err}"))?;
		let _ = sender.get_mut().stopped().await;
		Ok(())
	}
}
