// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use glam::UVec2;

use super::Vm;

impl Vm {
	/**
	 * Fills in all connected components or "holes" of the block map except the
	 * largest one.
	 *
	 * The purpose of this method is to prevent players from being physically
	 * separated by the blocks so they cannot fight each other and the game would
	 * be boring.
	 *
	 * This method works by performing flood fill on all non-block (non-positive)
	 * cells to identify the connected components (regions in which players
	 * aren't separated). Then the largest connected component is kept with all
	 * others being "filled" in by setting their value to be one (corresponding
	 * to being a block with the threshold).
	 *
	 * Note that if the world size is greater than the range of the blocks, some
	 * holes might be unnecessarily removed, and if the world size is smaller,
	 * players still might be able to be isolated.
	 */
	pub(super) fn fill_holes(&mut self) -> Result<(), String> {
		const BLOCK: u32 = u32::MAX;
		const UNASSIGNED: u32 = u32::MAX - 1;

		let bmap = Vm::get_top(&mut self.stack)?;

		let mut data = bmap.iter().map(|&x| if x > 0.0 { BLOCK } else { UNASSIGNED }).collect::<Box<[u32]>>();

		let mut stack = Vec::new();
		let mut i = 0;
		let mut component = 0;
		let mut blocks_per_component = Vec::new();

		assert!(data.len() == (self.grid_size.x * self.grid_size.y) as usize);

		for y in 0..self.grid_size.y {
			for x in 0..self.grid_size.x {
				if data[i] == UNASSIGNED {
					// Performs flood fill for the current component
					stack.push(UVec2::new(x, y));

					let mut blocks_in_component = 0;
					while let Some(pos) = stack.pop() {
						if pos.x >= self.grid_size.x || pos.y >= self.grid_size.y { continue; }
						let i = (pos.x + pos.y * self.grid_size.x) as usize;
						if data[i] == UNASSIGNED {
							data[i] = component;
							blocks_in_component += 1;
							stack.push(UVec2::new(pos.x.wrapping_sub(1), pos.y));
							stack.push(UVec2::new(pos.x + 1, pos.y));
							stack.push(UVec2::new(pos.x, pos.y.wrapping_sub(1)));
							stack.push(UVec2::new(pos.x, pos.y + 1));
						}
					}

					blocks_per_component.push(blocks_in_component);
					component += 1;
				}

				i += 1;
			}
		}

		let Some(largest_component) = blocks_per_component.into_iter().enumerate().max_by_key(|(_, n)| *n).map(|(i, _)| i as u32) else {
			return Ok(());
		};

		assert!(data.len() == bmap.len());

		for i in 0..data.len() {
			if data[i] != BLOCK && data[i] != largest_component {
				bmap[i] = 1.0;
			}
		}

		Ok(())
	}
}
