// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
pub mod config;
mod player_update_queue;
mod endpoint;
mod update;
pub mod scoring;
mod command;
pub mod joined;
mod lobby;
mod win_messages;

use std::str::FromStr;
use std::fmt::Write;
use std::collections::BTreeMap;
use std::{fs::{self as std_fs, File}, io::BufWriter};
use std::{mem, num::NonZeroUsize};
use std::net::{IpAddr, SocketAddr};
use std::time::{Instant, Duration};
use std::sync::{Arc, atomic::{AtomicUsize, Ordering}};

use ron::de;
use png::{Encoder, ColorType, BitDepth};
use tokio::{time, fs as tokio_fs, task, sync::{mpsc::Receiver, oneshot::Sender as OneshotSender}};
use quinn::VarInt;

use config::Config;
use player_update_queue::PlayerUpdateQueue;
use endpoint::{router::{Router, RouterEntry, GameServer}, ServerEndpoint, Repl};
use scoring::Scoring;
use lobby::Lobby;
use joined::JoinInfo;

use super::{
	message::{Message, MiscMessage}, message_stream::{DynMs, MsError}, connection_listener::DynConnLis,
	lan_discovery::{LanServerInfo, server::LanDiscoveryServer}, text_io::{self, TextIoString, ServerTextOutput}, timer::Timer,
};

use crate::{ServerArgs, LanDiscoveryConfig};
use crate::world::{
	World,
	event::{WorldEvent, EventChooser, move_chooser::{MoveChooser, PlayingMoveChooser, LobbyMoveChooser}, portal::PortalManager, effect_zone::EffectZones, flags::Flags},
	player::PlayerId, team::{Team, TeamId}, flag::Flag,
};
use crate::blocks::{Blocks, compressed::CompressedBlocks};
use crate::utils::{Clock, ctrl_c::CtrlCHandler};

pub const PORT: u16 = 4080; // UDP
const SANITY_CHECK_PERIOD: f32 = 0.25; // Four per second
pub(super) const SHUTTING_DOWN_MSG: &[u8] = b"server shutting down";
pub(super) const SHUTTING_DOWN_STATUS: VarInt = VarInt::from_u32(0);

pub(super) struct Server {
	clients: BTreeMap<PlayerId, Client>,
	connection_listener: Box<DynConnLis>,

	world: World,
	max_players: usize,
	instructions: Option<TextIoString>,
	event_chooser: EventChooser,
	playing_move_chooser: PlayingMoveChooser,
	scoring: Scoring,
	blocks: Blocks,
	compressed_blocks: CompressedBlocks,
	clock: Clock,

	lobby: Lobby,
	can_join_midgame: bool,
	timer: Option<Timer>,
	playing_timer: Option<Timer>,
	sync_timer: bool,
	team_info: TeamInfo,

	next_sanity_check: f32,
}

enum ServerMessage {
	LanDiscovery(OneshotSender<LanServerInfo>),
	AdminMessage(TextIoString),
}

pub(super) struct CertInfo {
	pub cert_path: Box<str>,
	pub key_path: Box<str>,
}

struct TeamInfo {
	map: BTreeMap<Arc<str>, TeamId>,
	shortest_aliases: Box<[Option<Arc<str>>]>,
}

impl TeamInfo {
	fn teams_command_output(&self, player_id: PlayerId, current_team: Option<TeamId>, world: &World) -> TextIoString {
		if world.has_teams() {
			/*
			 * Note that I can use `client.join_info.team` but this can be
			 * Some(...) if the player is spectating (the next team they will
			 * join).
			 *
			 * However, the number of players in each team that this command
			 * reports is the number of players in the world, not the number of
			 * clients (cannot easily do the latter as the borrow checker
			 * complains), so I want to keep things consistent.
			 */
			let mut s = String::new();
			for (index, freq) in world.team_freqs().into_iter().enumerate() {
				if index > 0 {
					s.push('\n');
				}

				let team_id = TeamId::from_index(index);
				s.push_str(world.get_team(team_id).unwrap().get_name());
				if let Some(Some(alias)) = self.shortest_aliases.get(index) {
					let _ = write!(s, " (alias = \"{alias}\")");
				}

				let _ = write!(s, ": {freq} player");
				if freq != 1 {
					s.push('s');
				}

				if Some(team_id) == current_team {
					if world.has_player(player_id) {
						s.push_str(" (including you)");
					} else {
						s.push_str(" (your team)");
					}
				}
			}

			TextIoString::new_truncate(s)
		} else {
			text_io::from_const!("Teams are not enabled.")
		}
	}
}

struct Client {
	stream: Box<DynMs>,
	to_remove: bool,
	spectating: Option<SpectatingMode>,
	join_info: JoinInfo,

	/*
	 * All per-player messages are sent in the order they are pushed.
	 *
	 * The usize for each per-player message is an index to a message in the
	 * all-players message list, with this per-player message being sent out
	 * before that all-players message.
	 */
	per_player_messages: Vec<(usize, Message)>,
	updates: PlayerUpdateQueue,
	allowed_chats: f32,
}

#[derive(Debug, Clone, Copy, PartialEq, Eq)]
enum SpectatingMode {
	/**
	 * The user explicitly chose to be spectating the game, either through
	 * running the `spectate` command or joining the game as a spectator.
	 */
	Manual,

	/**
	 * The user was put into spectator mode by being eliminated, or joined a game
	 * as a player where players cannot join mid-game.
	 */
	Automatic,
}

impl Client {
	fn new(stream: Box<DynMs>, spectating: Option<SpectatingMode>, join_info: JoinInfo) -> Client {
		Client {
			stream,
			to_remove: false,
			spectating,
			join_info,
			per_player_messages: Vec::new(),
			updates: PlayerUpdateQueue::new(),
			allowed_chats: CHAT_MAX_BURST,
		}
	}

	/**
	 * Inserts the given messages to the front of the per-player messages. Useful
	 * for ensuring that the `InitWorld` message is first received, which the
	 * client expects.
	 */
	fn set_init_per_player_messages(&mut self, messages: impl IntoIterator<Item = Message>) {
		self.per_player_messages.splice(..0, messages.into_iter().map(|msg| (0, msg))); // https://stackoverflow.com/a/47039490
	}

	fn send_messages(&mut self, messages: &[Message]) -> Result<(), MsError> {
		let mut per_player_messages = mem::take(&mut self.per_player_messages).into_iter().peekable();

		for (i, msg) in messages.iter().enumerate() {
			while let Some(ppm) = per_player_messages.next_if(|ppm| ppm.0 <= i) {
				self.stream.send(ppm.1)?;
			}

			self.stream.send(msg.clone())?;
		}


		// All per-player messages that are after the final all-players message
		for ppm in per_player_messages {
			self.stream.send(ppm.1)?;
		}

		self.stream.flush()
	}

	fn get_name(clients: &BTreeMap<PlayerId, Client>, id: PlayerId) -> &str {
		clients.get(&id).map_or("[unknown]", |client| &client.join_info.style.name)
	}
}

const CHAT_RATE_LIMIT: f32 = 2.0; // Chats per second
const CHAT_MAX_BURST: f32 = 10.0;

impl Server {
	pub(super) fn build(connection_listener: Box<DynConnLis>, config: Config, sanity_checks: bool) -> Result<Server, String> {
		config.validate().map_err(|err| format!("invalid config: {err}"))?;

		let block_map = config.blocks.generate_map().map_err(|err| format!("cannot generate block map: {err}"))?;
		let blocks = Blocks::new(config.blocks.block_size, config.blocks.grid_size, block_map.iter().map(|&x| x > 0.0).collect());
		let compressed_blocks = CompressedBlocks::from(&blocks);

		let custom_spawn_points = config.teams
			.iter()
			.enumerate()
			.filter_map(|(index, team)| team.spawn_point.map(|pos| (TeamId::from_index(index), pos)))
			.collect();

		let mut special_areas = config.special_areas.areas;
		let (teams, team_info, team_bases) = Config::get_teams(config.teams, &mut special_areas).map_err(|err| format!("invalid teams: {err}"))?;

		let can_join_midgame = config.scoring.can_join_midgame;

		let portal_manager = PortalManager::build(config.special_areas.portals, config.special_areas.portal_default_colours, &mut special_areas).map_err(|err| format!("portals: {err}"))?;
		let effect_zones = EffectZones::new(config.special_areas.effect_zones, &config.events, config.special_areas.effect_zone_default_colours, &mut special_areas);
		let scoring = Scoring::new(config.scoring, &teams, &mut special_areas);

		special_areas.retain(|area| !area.colour.fully_transparent()); // Fully transparent special areas are useless, so don't send them

		let flags = config.flags
			.iter()
			.map(|flag| Flag::new(flag.default_pos, flag.colour.get(teams.get(flag.team.index()).map(Team::get_colour))))
			.collect();

		let world = World::new(config.world, special_areas.into_boxed_slice(), teams, flags);
		let flags = Flags::new(config.flags, team_bases);

		let playing_move_chooser = PlayingMoveChooser::build(&world, &blocks, config.events.player_spawn_radius, custom_spawn_points)?;
		let lobby_move_chooser = LobbyMoveChooser::build(&world, &blocks, config.events.player_spawn_radius, config.lobby.local_spawn_radius)?;
		let event_chooser = EventChooser::new(portal_manager, effect_zones, flags, config.events);
		let lobby = Lobby::new(config.lobby, lobby_move_chooser);

		let playing_timer = config.timer;
		let timer = if lobby.active() { lobby.get_timer() } else { playing_timer.clone() };

		Ok(Server {
			clients: BTreeMap::new(),
			connection_listener,
			world,
			max_players: config.max_players.map_or(usize::MAX, NonZeroUsize::get),
			instructions: config.instructions,
			event_chooser,
			playing_move_chooser,
			scoring,
			blocks,
			compressed_blocks,
			clock: Clock::new_world(),
			lobby,
			can_join_midgame,
			timer,
			playing_timer,
			sync_timer: true,
			team_info,
			next_sanity_check: if sanity_checks { SANITY_CHECK_PERIOD } else { f32::INFINITY },
		})
	}

	async fn run(mut self, mut receiver: Receiver<ServerMessage>, lan_info: Option<LanServerInfo>, shared_player_count: Arc<AtomicUsize>) {
		const PERIOD: Duration = Duration::from_nanos(10_416_667); // 1/96 seconds

		let mut prev_time = Instant::now();
		let mut prev_player_count = 0;

		log::trace!(target: "perf", "period,{}", PERIOD.as_secs_f64());

		loop {
			let client_count = self.clients.len();
			shared_player_count.store(client_count, Ordering::Relaxed);

			let mut messages = Vec::new();
			while let Ok(msg) = receiver.try_recv() {
				match msg {
					ServerMessage::LanDiscovery(sender) => {
						if let Some(info) = &lan_info {
							let _ = sender.send(info.clone().with_cur_players(client_count.try_into().unwrap_or(u16::MAX)));
						}
					},
					ServerMessage::AdminMessage(msg) => messages.push(Message::Misc(MiscMessage::ServerTextOutput(ServerTextOutput::Admin(msg)))),
				}
			}

			let cur_time = Instant::now();
			let dt = cur_time.duration_since(prev_time).as_secs_f32();
			prev_time = cur_time;

			if self.update(dt, messages).is_err() {
				for client in self.clients.into_values() {
					client.stream.close();
				}

				return;
			}

			if prev_player_count != client_count {
				log::info!("client count = {client_count}");
				log::trace!(target: "perf", "client count,{client_count}"); // Slightly uglier format for the performance log
				prev_player_count = client_count;
			}

			let elapsed = cur_time.elapsed();
			let time = elapsed.as_secs_f64();
			log::trace!(target: "perf", "loop,{time}");

			time::sleep(PERIOD.saturating_sub(elapsed)).await;
		}
	}

	fn reset(&mut self) -> Vec<WorldEvent> {
		// Would like to de-duplicate this code with a method but the borrow checker doesn't allow that
		let move_chooser: &dyn MoveChooser = if self.lobby.active() { self.lobby.get_move_chooser() } else { &self.playing_move_chooser };
		self.event_chooser.reset_get_and_apply_events(&mut self.world, &self.blocks, move_chooser)
	}

	fn can_join(&self) -> bool {
		self.lobby.active() || self.can_join_midgame
	}

	fn set_timer(&mut self, timer: Option<Timer>) {
		self.timer = timer;
		self.sync_timer = true;
	}
}

fn deserialise_config(data: Vec<u8>) -> Result<Config, String> {
	de::from_bytes(&data).map_err(|err| format!("invalid config file: {err}"))
}

fn load_config(path: &str) -> Result<Config, String> {
	deserialise_config(std_fs::read(path).map_err(|err| format!("cannot read file \"{path}\": {err}"))?)
}

async fn load_config_async(path: &str) -> Result<Config, String> {
	deserialise_config(tokio_fs::read(path).await.map_err(|err| format!("cannot read file \"{path}\": {err}"))?)
}

#[tokio::main]
pub async fn run(args: ServerArgs) -> Result<(), String> {
	log::info!("Starting the server!");

	let addr = SocketAddr::new(IpAddr::from_str(&args.addr).map_err(|err| format!("invalid address: {err}"))?, args.port);
	let lan_discovery = if args.lan_discovery != LanDiscoveryConfig::Disabled {
		match LanDiscoveryServer::build(addr.ip()) {
			Ok(ld) => {
				log::info!("Created LAN discovery server");
				Some(ld)
			},
			Err(err) => {
				log::warn!("failed creating LAN discovery: {err}");
				None
			},
		}
	} else {
		None
	};

	let mut router = Router::default();
	for game in &args.game_servers {
		let entry = RouterEntry::build(game).map_err(|err| format!("failed parsing router entry: {err}"))?;
		let server = GameServer::build(addr, &entry, args.sanity_checks, lan_discovery.is_some()).await.map_err(|err| format!("failed creating game server: {err}"))?;
		router.add_game_server(entry, server).map_err(|err| format!("failed adding game server: {err}"))?;
	}

	tokio::spawn(async { CtrlCHandler::new().listen().await; });

	let cert_info = args.cert_path.as_ref().zip(args.key_path.as_ref()).map(|c| CertInfo { cert_path: c.0.clone(), key_path: c.1.clone() });
	let endpoint = ServerEndpoint::build(addr, cert_info, router, lan_discovery, args.lan_discovery).await.map_err(|err| format!("failed creating server endpoint: {err}"))?;

	let task = {
		let endpoint = Arc::clone(&endpoint);
		tokio::spawn(endpoint.run())
	};

	/*
	 * Need to run the REPL on the main thread as for whatever reason, `linefeed`
	 * doesn't work as well when on another thread. In particular, when pressing
	 * Ctrl+C on a line, the line doesn't get immediately get cancelled.
	 */
	let repl = Repl::new(Arc::clone(&endpoint), args, addr);
	task::block_in_place(move || repl.run());

	let _ = task.await;
	Ok(())
}

pub fn save_blocks(config_path: &str, output_path: &str) -> Result<(), String> {
	let config = load_config(config_path)?;
	let bmap = config.blocks.generate_map().map_err(|err| format!("cannot generate config: {err}"))?;
	let file = File::create(output_path).map_err(|err| format!("cannot open \"{output_path}\" for saving image: {err}"))?;

	let mut encoder = Encoder::new(BufWriter::new(file), config.blocks.grid_size.x, config.blocks.grid_size.y);
	encoder.set_color(ColorType::Grayscale);
	encoder.set_depth(BitDepth::Eight);

	let mut writer = encoder.write_header().map_err(|err| format!("cannot write header of image: {err}"))?;
	let mut output_data = Vec::with_capacity(config.blocks.grid_size.element_product() as usize);

	for y in (0..config.blocks.grid_size.y).rev() {
		let i = (y * config.blocks.grid_size.x) as usize;
		for &x in &bmap[i..i + config.blocks.grid_size.x as usize] {
			output_data.push(if x > 0.0 { 0xff } else { 0x00 });
		}
	}

	writer.write_image_data(&output_data).map_err(|err| format!("cannot save image to \"{output_path}\": {err}"))?;
	writer.finish().map_err(|err| format!("cannot flush image data: {err}"))
}
