// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
#[cfg(test)]
mod tests;

use std::io::{Read, Result as IoResult, Error as IoError, ErrorKind};
use std::fmt::{Display, Formatter, Result as FmtResult};

use arrayvec::ArrayVec;
use tokio::io::{AsyncReadExt, AsyncWriteExt};

/**
 * A variable-length unsigned integer that supports values from 0 to 0x4040403f
 * (a bit over 2^30) inclusive.
 *
 * This is called "Vu30" as it's of similar size to a hypothetical "u30" type
 * and it's variable length.
 *
 * This integer consists of 1 to 4 bytes with larger numbers requiring more
 * bytes to be represented.
 *
 * The first byte consists of llxx xxxx, where ll specifies the number of
 * remaining bytes (0 – 3) and the remaining bits are the least significant bits
 * of the number.
 *
 * All remaining bytes are of increasing significance, similar to little-endian.
 */
#[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub struct Vu30(u32);

impl Vu30 {
	pub const ZERO: Vu30 = Vu30(0);

	const B2_BEGIN: u32 = 1 << 6;
	const B3_BEGIN: u32 = Vu30::B2_BEGIN + (1 << (6 + 8));
	const B4_BEGIN: u32 = Vu30::B3_BEGIN + (1 << (6 + 8 * 2));
	pub const MAX: Vu30 = Vu30(Vu30::B4_BEGIN + (1 << (6 + 8 * 3)) - 1);

	pub const fn from_u8(x: u8) -> Vu30 {
		Vu30(x as u32)
	}

	pub const fn try_from_u32(x: u32) -> Option<Vu30> {
		if x <= Vu30::MAX.0 { Some(Vu30(x)) }
		else { None }
	}

	pub const fn try_from_usize(x: usize) -> Option<Vu30> {
		if x <= Vu30::MAX.0 as usize { Some(Vu30(x as u32)) }
		else { None }
	}

	pub const fn get(self) -> u32 {
		self.0
	}

	/// Returns the length when serialised.
	pub const fn len(self) -> usize {
		if self.0 < Vu30::B2_BEGIN { 1 }
		else if self.0 < Vu30::B3_BEGIN { 2 }
		else if self.0 < Vu30::B4_BEGIN { 3 }
		else { 4 }
	}
}

impl Display for Vu30 {
	fn fmt(&self, fmt: &mut Formatter<'_>) -> FmtResult {
		write!(fmt, "{}", self.0)
	}
}

pub enum ReadBytesError {
	TooLong(usize),
	Io(IoError),
}

impl From<IoError> for ReadBytesError {
	fn from(error: IoError) -> ReadBytesError {
		ReadBytesError::Io(error)
	}
}

impl Display for ReadBytesError {
	fn fmt(&self, fmt: &mut Formatter<'_>) -> FmtResult {
		match self {
			ReadBytesError::Io(err) => write!(fmt, "{err}"),
			ReadBytesError::TooLong(size) => write!(fmt, "data too long, containing {size} byte(s)"),
		}
	}
}

impl From<ReadBytesError> for String {
	fn from(error: ReadBytesError) -> String {
		error.to_string()
	}
}

pub trait ReadVu30: Read {
	// Can't seem to deduplicate this code between the async version annoyingly
	fn read_vu30(&mut self) -> IoResult<Vu30> {
		let first = self.read_u8()?;
		match first & 0xc0 {
			0x00 => Ok(Vu30(first as u32)),
			0x40 => {
				let next = self.read_u8()?;
				Ok(Vu30(Vu30::B2_BEGIN + (((next as u32) << 6) | (first & 0x3f) as u32)))
			},
			0x80 => {
				let next = self.read_u16_le()?;
				Ok(Vu30(Vu30::B3_BEGIN + (((next as u32) << 6) | (first & 0x3f) as u32)))
			},
			_ /* 0xc0 */ => {
				let mut next = [0; 3];
				self.read_exact(&mut next)?;
				let next_u24_le = next[0] as u32 | ((next[1] as u32) << 8) | ((next[2] as u32) << 16);
				Ok(Vu30(Vu30::B4_BEGIN + ((next_u24_le << 6) | (first & 0x3f) as u32)))
			},
		}
	}

	fn read_u8(&mut self) -> IoResult<u8> {
		let mut buf = [0; 1];
		self.read_exact(&mut buf)?;
		Ok(buf[0])
	}

	fn read_u16_le(&mut self) -> IoResult<u16> {
		let mut buf = [0; 2];
		self.read_exact(&mut buf)?;
		Ok((buf[0] as u16) | ((buf[1] as u16) << 8))
	}
}

impl<R> ReadVu30 for R where R: Read {}

pub trait AsyncReadVu30: AsyncReadExt + Unpin {
	async fn read_vu30(&mut self) -> IoResult<Vu30> {
		let first = self.read_u8().await?;
		match first & 0xc0 {
			0x00 => Ok(Vu30(first as u32)),
			0x40 => {
				let next = self.read_u8().await?;
				Ok(Vu30(Vu30::B2_BEGIN + (((next as u32) << 6) | (first & 0x3f) as u32)))
			},
			0x80 => {
				let next = self.read_u16_le().await?;
				Ok(Vu30(Vu30::B3_BEGIN + (((next as u32) << 6) | (first & 0x3f) as u32)))
			},
			_ /* 0xc0 */ => {
				let mut next = [0; 3];
				self.read_exact(&mut next).await?;
				let next_u24_le = next[0] as u32 | ((next[1] as u32) << 8) | ((next[2] as u32) << 16);
				Ok(Vu30(Vu30::B4_BEGIN + ((next_u24_le << 6) | (first & 0x3f) as u32)))
			},
		}
	}

	/**
	 * Reads a Vu30 for a length and then reads data up to that length which is
	 * appended to the `bytes` Vec<u8> supplied. Allows setting a size limit to
	 * avoid application layer denial of service attacks.
	 */
	async fn read_into_bytes(&mut self, bytes: &mut Vec<u8>, size_limit: usize) -> Result<(), ReadBytesError> {
		let len = self.read_vu30().await?.get() as usize;
		if len > size_limit {
			return Err(ReadBytesError::TooLong(len));
		}

		bytes.reserve(len);
		let mut reader = self.take(len as u64);
		let count = reader.read_to_end(bytes).await?;

		debug_assert!(count <= len);
		if count < len {
			return Err(ReadBytesError::Io(IoError::new(ErrorKind::UnexpectedEof, "failed reading all bytes, got EOF")));
		}

		Ok(())
	}

	/**
	 * A wrapper over `read_into_bytes` which creates and returns a new Vec.
	 */
	async fn read_bytes(&mut self, size_limit: usize) -> Result<Vec<u8>, ReadBytesError> {
		let mut bytes = Vec::new();
		self.read_into_bytes(&mut bytes, size_limit).await?;
		Ok(bytes)
	}

	/**
	 * Similar to `read_bytes` but requires the data to be valid UTF-8.
	 */
	#[cfg(feature = "client")]
	async fn read_string(&mut self, size_limit: usize) -> Result<String, String> {
		let bytes = self.read_bytes(size_limit).await?;
		String::from_utf8(bytes).map_err(|err| format!("invalid UTF-8: {err}"))
	}
}

impl<R> AsyncReadVu30 for R where R: AsyncReadExt + Unpin {}

pub trait AsyncWriteVu30: AsyncWriteExt + Unpin {
	async fn write_vu30(&mut self, x: Vu30) -> IoResult<()> {
		let mut data = ArrayVec::new();
		data.extend_vu30(x);
		self.write_all(&data).await
	}

	async fn write_bytes(&mut self, data: &[u8]) -> Result<(), String> {
		let failed_writing = |err| format!("failed writing data: {err}");
		self.write_vu30(Vu30::try_from_usize(data.len()).ok_or_else(super::vu30_too_large)?).await.map_err(failed_writing)?;
		self.write_all(data).await.map_err(failed_writing)
	}
}

impl<W> AsyncWriteVu30 for W where W: AsyncWriteExt + Unpin {}

pub trait ExtendVu30 {
	fn extend_vu30(&mut self, mut x: Vu30) {
		if x.0 < Vu30::B2_BEGIN { // 1 byte
			self.push_u8(x.0 as u8);
		} else if x.0 < Vu30::B3_BEGIN { // 2 bytes
			x.0 -= Vu30::B2_BEGIN;
			let first = 0x40 | (x.0 & 0x3f) as u8;
			let next = (x.0 >> 6) as u8;
			self.push_u8(first);
			self.push_u8(next);
		} else if x.0 < Vu30::B4_BEGIN { // 3 bytes
			x.0 -= Vu30::B3_BEGIN;
			let first = 0x80 | (x.0 & 0x3f) as u8;
			let next = (x.0 >> 6) as u16;
			self.push_u8(first);
			self.push_u16(next);
		} else { // 4 bytes
			x.0 -= Vu30::B4_BEGIN;
			let first = 0xc0 | (x.0 & 0x3f) as u8;
			let next = x.0 >> 6;
			self.push_u8(first);
			self.push_u16((next & 0xffff) as u16);
			self.push_u8((next >> 16) as u8);
		}
	}

	fn push_u8(&mut self, x: u8);
	fn push_u16(&mut self, x: u16);
}

impl ExtendVu30 for Vec<u8> {
	fn push_u8(&mut self, x: u8) {
		self.push(x);
	}

	fn push_u16(&mut self, x: u16) {
		self.extend_from_slice(&x.to_le_bytes());
	}
}

impl ExtendVu30 for ArrayVec<u8, 4> {
	fn push_u8(&mut self, x: u8) {
		self.push(x);
	}

	fn push_u16(&mut self, x: u16) {
		self.try_extend_from_slice(&x.to_le_bytes()).unwrap();
	}
}
