// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use quinn::SendStream;
use tokio::io::{Error as IoError, BufWriter, AsyncWriteExt};

use super::{Vu30, SERVER_MAGIC_NUMBER, vu30::AsyncWriteVu30};

// Note that the strings here can be empty
pub enum Response {
	/// The request was successful. Includes the index of the successful
	/// subrequest.
	Success(Vu30),

	/// The request was understood by the server but couldn't be fulfilled.
	/// Contains an index of the subrequest and an error message.
	Understood(Vu30, String),

	/// A general-purpose status for errors that don't fall in the other
	/// categories.
	#[allow(unused)] // Right now not yet constructed, this is fine
	Error(String),

	/// The client violated the SERP protocol.
	Invalid(String),

	/// A subrequest (index provided) that the client sent contained an invalid
	/// payload.
	BadPayload(Vu30, String),

	/// The server didn't want to process the request as it was too long.
	/// Includes the maximum size the server wants and an additional error
	/// message.
	TooLong(Vu30, String),

	/// The server isn't familiar with the request types of all subrequests.
	UnsupportedRequest(String),

	/// A version of a subrequest sent by the client isn't supported. A version
	/// of that request that the server supports is included in the response.
	UnsupportedVersion {
		index: Vu30,
		supported_version: Vu30,
		message: String,
	},
}

pub(super) mod statuses {
	use super::Vu30;

	pub const SUCCESS: Vu30 = Vu30::from_u8(0);
	pub const UNDERSTOOD: Vu30 = Vu30::from_u8(1);
	pub const ERROR: Vu30 = Vu30::from_u8(2);
	pub const INVALID: Vu30 = Vu30::from_u8(3);
	pub const BAD_PAYLOAD: Vu30 = Vu30::from_u8(4);
	pub const TOO_LONG: Vu30 = Vu30::from_u8(5);
	pub const UNSUPPORTED_REQUEST: Vu30 = Vu30::from_u8(6);
	pub const UNSUPPORTED_VERSION: Vu30 = Vu30::from_u8(7);
}

fn failed_writing_status(err: IoError) -> String {
	format!("failed writing status: {err}")
}


fn failed_writing_index(err: IoError) -> String {
	format!("failed writing subrequest index: {err}")
}

impl Response {
	pub async fn reply(&self, sender: &mut BufWriter<SendStream>) -> Result<(), String> {
		sender.write_all(SERVER_MAGIC_NUMBER).await.map_err(|err| format!("failed writing magic number: {err}"))?;

		match self {
			Response::Success(index) => {
				sender.write_vu30(statuses::SUCCESS).await.map_err(failed_writing_status)?;
				sender.write_vu30(*index).await.map_err(failed_writing_index)?;
			},
			Response::Understood(index, msg) => Response::respond_vu30_string(sender, statuses::UNDERSTOOD, *index, msg).await?,
			Response::Error(msg) => Response::respond_string(sender, statuses::ERROR, msg).await?,
			Response::Invalid(msg) => Response::respond_string(sender, statuses::INVALID, msg).await?,
			Response::BadPayload(index, msg) => Response::respond_vu30_string(sender, statuses::BAD_PAYLOAD, *index, msg).await?,
			Response::TooLong(max_size, msg) => Response::respond_vu30_string(sender, statuses::TOO_LONG, *max_size, msg).await?,
			Response::UnsupportedRequest(msg) => Response::respond_string(sender, statuses::UNSUPPORTED_REQUEST, msg).await?,
			Response::UnsupportedVersion { index, supported_version, message } => {
				sender.write_vu30(statuses::UNSUPPORTED_VERSION).await.map_err(failed_writing_status)?;
				sender.write_vu30(*index).await.map_err(failed_writing_index)?;
				sender.write_vu30(*supported_version).await.map_err(|err| format!("failed writing supported version: {err}"))?;
				sender.write_bytes(message.as_bytes()).await?;
			},
		}

		sender.flush().await.map_err(|err| format!("failed flushing sender: {err}"))?;

		if !matches!(self, Response::Success(_)) {
			let sender = sender.get_mut();
			let _ = sender.finish();
			let _ = sender.stopped().await;
		}

		Ok(())
	}

	async fn respond_string(sender: &mut BufWriter<SendStream>, status: Vu30, msg: &str) -> Result<(), String> {
		sender.write_vu30(status).await.map_err(failed_writing_status)?;
		sender.write_bytes(msg.as_bytes()).await
	}

	async fn respond_vu30_string(sender: &mut BufWriter<SendStream>, status: Vu30, num: Vu30, msg: &str) -> Result<(), String> {
		sender.write_vu30(status).await.map_err(failed_writing_status)?;
		sender.write_vu30(num).await.map_err(|err| format!("failed writing vu30: {err}"))?;
		sender.write_bytes(msg.as_bytes()).await
	}
}
