// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::{str, fmt::{Display, Formatter, Result as FmtResult}};

use quinn::{Connection, SendStream, RecvStream};
use tokio::{io::{BufReader, BufWriter, AsyncReadExt}, time::{self, Duration}};
use serde::{Serialize, Deserialize};
use bincode::Options;
#[cfg(feature = "client")] use tokio::io::AsyncWriteExt;

use super::{Response, Vu30, SIZE_LIMIT, CLIENT_MAGIC_NUMBER, vu30::{AsyncReadVu30, ReadVu30, ReadBytesError}};
#[cfg(feature = "client")] use super::{SERVER_MAGIC_NUMBER, vu30::{AsyncWriteVu30, ExtendVu30}, response::statuses};
use super::super::message_stream::quic_common;

use crate::world::player::PlayerStyle;

pub enum Request {
	Play(PlayRequest),
	Info,
}

#[derive(Serialize, Deserialize)]
pub struct PlayRequest {
	pub game_id: Box<str>,
	pub spectate: bool,
	pub style: PlayerStyle,
	pub extension: Box<[u8]>,
}

pub const PLAY_VERSION_FIRST_DEV: u32 = 0x20000000;
//pub const PLAY_VERSION: Vu30 = Vu30::try_from_u32(PLAY_VERSION_FIRST_DEV + 46).unwrap(); // Use this for the next development version
pub const PLAY_VERSION: Vu30 = Vu30::try_from_u32(4).unwrap();
pub const INFO_VERSION: Vu30 = Vu30::ZERO;

pub mod request_types { // Would be nice if I could more easily do this and not need to worry about these being in order
	use super::Vu30;

	pub const PLAY: Vu30 = Vu30::from_u8(0);
	pub const INFO: Vu30 = Vu30::from_u8(1);
}

pub enum RequestError {
	Response(Response), // The server returned a valid SERP error response
	Other(String), // Any other error
}

impl From<Response> for RequestError {
	fn from(response: Response) -> RequestError {
		RequestError::Response(response)
	}
}

impl From<String> for RequestError {
	fn from(err: String) -> RequestError {
		RequestError::Other(err)
	}
}

impl Display for RequestError {
	fn fmt(&self, fmt: &mut Formatter<'_>) -> FmtResult {
		match self {
			RequestError::Response(response) => {
				write!(fmt, "server response: ")?;
				match response {
					Response::Success(_) => write!(fmt, "success"),
					Response::Understood(_, msg) => write!(fmt, "understod request but failed: {msg}"),
					Response::Error(msg) => write!(fmt, "error: {msg}"),
					Response::Invalid(msg) => write!(fmt, "server failed processing request: {msg}"), // The request might not actually be invalid, the server can just lie
					Response::BadPayload(_, msg) => write!(fmt, "bad payload: {msg}"),
					Response::TooLong(size_limit, msg) => write!(fmt, "message was too long, server says it violated size limit of {size_limit}: {msg}"),
					Response::UnsupportedRequest(msg) => write!(fmt, "unsupported request: {msg}"),
					Response::UnsupportedVersion { index: _, supported_version, message } => {
						write!(fmt, "unsupported version, server supports version {supported_version}")?;
						if supported_version.get() >= PLAY_VERSION_FIRST_DEV {
							write!(fmt, " (which is a development version)")?;
						}
						write!(fmt, ": {message}")
					},
				}
			},
			RequestError::Other(err) => write!(fmt, "{err}"),
		}
	}
}

impl Request {
	/**
	 * Sends a single request.
	 */
	#[cfg(feature = "client")]
	pub async fn send(&self, connection: &Connection) -> Result<(BufWriter<SendStream>, BufReader<RecvStream>), RequestError> {
		let mut request_data = Vec::new();
		let (request_type, request_version) = match self {
			Request::Play(_) => (request_types::PLAY, PLAY_VERSION),
			Request::Info => (request_types::INFO, INFO_VERSION),
		};

		request_data.extend_vu30(request_type);
		request_data.extend_vu30(request_version);

		match self {
			Request::Play(request) => {
				let data = quic_common::bincode_options().with_limit(SIZE_LIMIT as u64).serialize(&request).map_err(|err| format!("failed serialising Play request payload: {err}"))?;
				request_data.extend_vu30(Vu30::try_from_usize(data.len()).ok_or_else(super::vu30_too_large)?);
				request_data.extend_from_slice(&data);
			},
			Request::Info => request_data.extend_vu30(Vu30::ZERO), // No payload
		}

		let (mut sender, mut receiver) = {
			let (sender, receiver) = connection.open_bi().await.map_err(|err| format!("failed creating bidirectional stream: {err}"))?;
			(BufWriter::new(sender), BufReader::new(receiver))
		};

		sender.write_all(CLIENT_MAGIC_NUMBER).await.map_err(|err| format!("failed writing magic number: {err}"))?;
		sender.write_vu30(Vu30::try_from_usize(request_data.len()).ok_or_else(super::vu30_too_large)?).await.map_err(|err| format!("failed writing request length: {err}"))?;
		sender.write_all(&request_data).await.map_err(|err| format!("failed writing request: {err}"))?;
		sender.flush().await.map_err(|err| format!("failed flushing sender stream: {err}"))?;

		let mut buf = [0; const { SERVER_MAGIC_NUMBER.len() }];
		receiver.read_exact(&mut buf).await.map_err(|err| format!("failed reading magic number: {err}"))?;
		if buf != SERVER_MAGIC_NUMBER {
			return Err(RequestError::Other(format!("invalid magic number, got {buf:?}")));
		}

		let status = receiver.read_vu30().await.map_err(|err| format!("failed reading status: {err}"))?;
		match status {
			statuses::SUCCESS => {
				Request::read_subrequest_index(&mut receiver).await?;
				Ok((sender, receiver))
			},
			statuses::UNDERSTOOD => {
				let index = Request::read_subrequest_index(&mut receiver).await?;
				let msg = receiver.read_string(SIZE_LIMIT).await?;
				Err(RequestError::Response(Response::Understood(index, msg)))
			},
			statuses::ERROR => {
				let msg = receiver.read_string(SIZE_LIMIT).await?;
				Err(RequestError::Response(Response::Error(msg)))
			},
			statuses::INVALID => {
				let msg = receiver.read_string(SIZE_LIMIT).await?;
				Err(RequestError::Response(Response::Invalid(msg)))
			},
			statuses::BAD_PAYLOAD => {
				let index = Request::read_subrequest_index(&mut receiver).await?;
				let msg = receiver.read_string(SIZE_LIMIT).await?;
				Err(RequestError::Response(Response::BadPayload(index, msg)))
			},
			statuses::TOO_LONG => {
				let max_size = receiver.read_vu30().await.map_err(|err| format!("failed reading server size limit: {err}"))?;
				let msg = receiver.read_string(SIZE_LIMIT).await?;
				Err(RequestError::Response(Response::TooLong(max_size, msg)))
			},
			statuses::UNSUPPORTED_REQUEST => {
				let msg = receiver.read_string(SIZE_LIMIT).await?;
				Err(RequestError::Response(Response::UnsupportedRequest(msg)))
			},
			statuses::UNSUPPORTED_VERSION => {
				let index = Request::read_subrequest_index(&mut receiver).await?;
				let supported_version = receiver.read_vu30().await.map_err(|err| format!("failed reading supported version: {err}"))?;
				let msg = receiver.read_string(SIZE_LIMIT).await?;
				Err(RequestError::Response(Response::UnsupportedVersion { index, supported_version, message: msg }))
			},
			_ => Err(RequestError::Other(format!("unknown response status {}", status.get()))),
		}
	}

	/**
	 * Reads the subrequest index in the stream and returns it. This must be zero
	 * and an error is produced if it isn't. In the general case when multiple
	 * requests can be sent, it's possible for this to not be zero.
	 */
	#[cfg(feature = "client")]
	async fn read_subrequest_index(receiver: &mut BufReader<RecvStream>) -> Result<Vu30, String> {
		let index = receiver.read_vu30().await.map_err(|err| format!("failed reading subrequest index: {err}"))?;
		if index != Vu30::ZERO {
			return Err(String::from("subrequest index out of bounds, expected zero"));
		}
		Ok(index)
	}

	/**
	 * Receives a request from the client.
	 *
	 * Returns the request, its index and two ends of the bidirectional stream on
	 * success. Returns a response to send back on failure, or a string if it's
	 * some other error. For example, if the magic number doesn't match, it
	 * probably isn't a Spaceships client so there's no point sending an actual
	 * response.
	 */
	pub async fn receive(connection: &Connection) -> Result<(Request, Vu30, BufWriter<SendStream>, BufReader<RecvStream>), Result<(Response, BufWriter<SendStream>), String>> {
		time::timeout(Duration::from_secs(15), async {
			let (sender, mut receiver) = {
				let (sender, receiver) = connection.accept_bi().await.map_err(|err| Err(format!("failed accepting bidirectional stream from client: {err}")))?;
				(BufWriter::new(sender), BufReader::new(receiver))
			};

			match Request::receive_from_receiver(&mut receiver).await {
				Ok((request, index)) => Ok((request, index, sender, receiver)),
				Err(Ok(response)) => Err(Ok((response, sender))),
				Err(Err(err)) => Err(Err(err)),
			}
		}).await.unwrap_or_else(|_| Err(Err(String::from("reading request timed out"))))
	}

	async fn receive_from_receiver(receiver: &mut BufReader<RecvStream>) -> Result<(Request, Vu30), Result<Response, String>> {
		let mut buf = [0; const { CLIENT_MAGIC_NUMBER.len() }];
		receiver.read_exact(&mut buf).await.map_err(|err| Err(format!("failed reading magic number: {err}")))?;
		if buf != CLIENT_MAGIC_NUMBER {
			return Err(Err(format!("invalid magic number, got {buf:?}")));
		}

		let buf = match receiver.read_bytes(SIZE_LIMIT).await {
			Ok(buf) => buf,
			Err(ReadBytesError::TooLong(_)) => return Err(Ok(Response::TooLong(Vu30::try_from_usize(SIZE_LIMIT).unwrap_or(Vu30::MAX), String::from("request too large")))),
			Err(ReadBytesError::Io(_)) => return Err(Ok(Response::Invalid(String::from("failed reading request")))),
		};

		let mut reader = buf.as_slice();
		let request_type = ReadVu30::read_vu30(&mut reader).map_err(|_| Ok(Response::Invalid(String::from("failed reading request type"))))?;
		let request_version = ReadVu30::read_vu30(&mut reader).map_err(|_| Ok(Response::Invalid(String::from("failed reading request version"))))?;
		let subrequest_len = ReadVu30::read_vu30(&mut reader).map_err(|_| Ok(Response::Invalid(String::from("failed reading request length"))))?.get() as usize;

		let request = match request_type {
			request_types::PLAY => {
				if request_version != PLAY_VERSION {
					return Err(Ok(Response::UnsupportedVersion { index: Vu30::ZERO, supported_version: PLAY_VERSION, message: format!("only version {PLAY_VERSION} supported") }));
				}

				let Some((data, _)) = reader.split_at_checked(subrequest_len) else {
					return Err(Ok(Response::Invalid(String::from("failed reading subrequest data, too long"))));
				};

				Request::Play(quic_common::bincode_options().with_limit(SIZE_LIMIT as u64).deserialize(data).map_err(|_| Ok(Response::Invalid(String::from("failed deserialising Play request"))))?)
			},
			request_types::INFO => {
				if request_version != INFO_VERSION {
					return Err(Ok(Response::UnsupportedVersion { index: Vu30::ZERO, supported_version: INFO_VERSION, message: format!("only version {INFO_VERSION} supported") }));
				}

				if subrequest_len > 0 {
					return Err(Ok(Response::BadPayload(Vu30::ZERO, String::from("expected no payload"))));
				}

				Request::Info
			},
			_ => return Err(Ok(Response::UnsupportedRequest(format!("only requests supported are Play (version {PLAY_VERSION}) and Info (version {INFO_VERSION})")))),
		};

		Ok((request, Vu30::ZERO))
	}
}
