// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
/**
 * A module for the Spaceships Extensible Request Protocol (SERP).
 *
 * This is an application layer protocol over QUIC that handles requests made by
 * clients to the server, and the server's corresponding response.
 *
 * This protocol is designed to be forward compatible by providing the ability
 * to send multiple subrequests in a single request. This allows the client to
 * say send version 0 of the Info request as well as version 1 to receive
 * information about the server. If the server is new and supports version 1 of
 * the Info request, it can and likely will respond with that version of the
 * request which likely will contain more data. If the server doesn't support
 * version 1, they can still understand the request for version 0 and send a
 * corresponding response.
 *
 * I also designed this protocol to gracefully support the case that this game
 * gets forked. If that does happen and the developer(s) of that fork want to
 * extend the protocol, they can use their own request numbers to implement
 * their custom requests and avoid conflict with other versions of the game.
 *
 * This will be relevant for some requests like the Play request, which if that
 * protocol is changed in a fork, choosing the next version number won't be
 * appropriate as potentially weird behaviour would arise once the other game
 * updates to that version and two different protocols would share the same
 * version. To solve this, I'm proposing a central registry of allocation of
 * blocks of request numbers for each fork, to avoid conflicts in the request
 * numbers.
 *
 * A core part of this protocol is the Vu30 type, which stands for
 * variable-length unsigned 30-bit integer. This can store integers ranging from
 * 0 to a bit over 2^30, with each integer taking up 1, 2, 3 or 4 bytes. The
 * exact details of this format are specified in the vu30 module.
 *
 * Vu30s are used in the request type, request version as well as lengths.
 *
 * Now here is the format of the request and response in pseudocode:
 *
 * struct Request {
 * 	magic_number: \[u8\] = CLIENT_MAGIC_NUMBER,
 *
 * 	// Remaining length of the data handled by SERP.
 * 	header_length: Vu30,
 * 	subrequests: \[SubRequest\],
 * }
 *
 * struct Subrequest {
 * 	// Request type, maintained by an individual or organised group of
 * 	// developers.
 * 	type: Vu30,
 * 	version: Vu30, // Version of that request
 *
 * 	// The length of a possible payload to this request, can be zero.
 * 	length: Vu30,
 *
 * 	// Arbitrary data with the format of that data specific to the request.
 * 	data: Vu30,
 * }
 *
 * struct Response {
 * 	magic_number: \[u8\] = SERVER_MAGIC_NUMBER,
 * 	status: Vu30,
 * 	// Remaining data depends on that status
 * }
 */
pub mod vu30;
pub mod request;
mod response;

pub use vu30::Vu30;
pub use request::{Request, PlayRequest};
pub use response::Response;

/*
 * The purpose of using two different magic numbers for the client and server is
 * to prevent things from possibly seeming valid if the client enters the
 * address of a QUIC echo server (responds what it receives).
 */
const CLIENT_MAGIC_NUMBER: &[u8] = b"seRp"; // 'R' for "request" is capitalised
const SERVER_MAGIC_NUMBER: &[u8] = b"Serp"; // 'S' for "server" is capitalised

pub const SIZE_LIMIT: usize = 1 << 16;

fn vu30_too_large() -> String {
	String::from("vu30 too large (seriously, what the fuck are you doing?)")
}
