// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::num::NonZeroUsize;
use std::sync::mpsc::{self, SyncSender, Receiver, TrySendError, TryRecvError};
use std::time::{Instant, Duration};

use glam::{UVec2, Vec2};

use super::{MessageStream, MsError, Security};

use crate::net::message::Message;
use crate::net::server::{Server, config::{Config, LanDiscoveryConfig, BlockConfig, EventConfig, SpecialAreaConfig, ScoringConfig, LobbyConfig, Op}};
use crate::net::connection_listener::singleplayer::SingleplayerConnectionListener;
use crate::net::serp::PlayRequest;
use crate::net::timer::Timer;
use crate::net::text_io;
use crate::world::{MAX_STEPS, config::WorldConfig};

pub(in crate::net) struct LocalMessageStream {
	sender: SyncSender<Message>,
	receiver: Receiver<Message>,
}

impl LocalMessageStream {
	pub fn new_pair() -> (LocalMessageStream, LocalMessageStream) {
		let ((sender_a, receiver_b), (sender_b, receiver_a)) = (mpsc::sync_channel(MAX_STEPS), mpsc::sync_channel(MAX_STEPS));
		(
			LocalMessageStream { sender: sender_a, receiver: receiver_a },
			LocalMessageStream { sender: sender_b, receiver: receiver_b },
		)
	}
}

impl MessageStream for LocalMessageStream {
	fn send(&mut self, msg: Message) -> Result<(), MsError> {
		self.sender.try_send(msg).map_err(|err| match err {
			TrySendError::Full(_) => MsError::Other(String::from("sync sender is full")),
			TrySendError::Disconnected(_) => MsError::Disconnected(None),
		})
	}

	fn receive(&mut self) -> Result<Option<Message>, MsError> {
		match self.receiver.try_recv() {
			Ok(msg) => Ok(Some(msg)),
			Err(TryRecvError::Empty) => Ok(None),
			Err(TryRecvError::Disconnected) => Err(MsError::Disconnected(None)),
		}
	}

	fn flush(&mut self) -> Result<(), MsError> {
		Ok(())
	}

	fn is_networked(&self) -> bool {
		false
	}

	fn source(&self) -> String {
		String::from("singleplayer")
	}

	fn ping(&self) -> Duration {
		Duration::ZERO
	}

	fn security(&self) -> Security {
		Security::Secure
	}
}

pub struct SingleplayerMessageStream {
	stream: LocalMessageStream,
	server: Server,
	prev_time: Instant,
}

impl SingleplayerMessageStream {
	pub fn new(request: PlayRequest, sanity_checks: bool) -> (Message, SingleplayerMessageStream) {
		let (stream_a, stream_b) = LocalMessageStream::new_pair();
		let config = Config {
			max_players: NonZeroUsize::new(8),
			instructions: Some(text_io::from_const!("Welcome to singleplayer! Right now this game mode isn't too interesting. Press backtick (or whatever you bound it to) to leave.")),
			world: WorldConfig {
				size: Vec2::INFINITY,
				friendly_fire: false,
			},
			lan_discovery: LanDiscoveryConfig::default(),
			blocks: BlockConfig {
				block_size: 3.try_into().unwrap(),
				grid_size: UVec2::splat(256),
				ops: [
					Op::WhiteNoise { seed: 0 },
					Op::Constant(0.9),
					Op::Sub,
				].into(),
			},
			events: EventConfig::default(),
			special_areas: SpecialAreaConfig::default(),
			scoring: ScoringConfig::default(),
			lobby: LobbyConfig::default(),
			timer: Some(Timer::count_up_forever()),
			teams: Vec::new(),
			flags: Vec::new(),
		};

		// Should be reasonable to assume creating the server never fails as right now this doesn't depend on user input
		let mut server = Server::build(Box::new(SingleplayerConnectionListener::new(stream_b, request)), config, sanity_checks).unwrap();
		server.update(0.0, Vec::new()).unwrap(); // Connects the client to the server
		let init_msg = stream_a.receiver.try_recv().unwrap();
		let stream = SingleplayerMessageStream { stream: stream_a, server, prev_time: Instant::now() };
		(init_msg, stream)
	}
}

impl MessageStream for SingleplayerMessageStream {
	fn send(&mut self, msg: Message) -> Result<(), MsError> { self.stream.send(msg) }
	fn receive(&mut self) -> Result<Option<Message>, MsError> { self.stream.receive() }

	fn flush(&mut self) -> Result<(), MsError> {
		let cur_time = Instant::now();
		let dt = cur_time.duration_since(self.prev_time).as_secs_f32();
		self.prev_time = cur_time;
		self.server.update(dt, Vec::new()).unwrap(); // The server should never randomly close
		self.stream.flush()
	}

	fn is_networked(&self) -> bool {
		false
	}

	fn ping(&self) -> Duration {
		Duration::ZERO
	}

	fn security(&self) -> Security {
		Security::Secure
	}
}
