// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
#[cfg(feature = "client")] use std::time::Duration;

use quinn::{Connection, SendStream, RecvStream};
use tokio::io::{BufReader, BufWriter};

use super::{MessageStream, DynMs, MsError, quic_common::{SenderTask, ReceiverTask, ReceiverTaskBuilder, ErrorContainer, MessageSer, MessageDe}};
#[cfg(feature = "client")] use super::Security;

use crate::net::{
	message::{Message, WorldMessage, MiscMessage},
	server::{SHUTTING_DOWN_MSG, SHUTTING_DOWN_STATUS},
	serp::{Response, Vu30}, connection_listener::IncomingConnection, utils::SocketAddrToCanonical,
	text_io::ClientTextInput,
};

use crate::world::player::update::{PlayerUpdate, PlayerBulkUpdate};

pub struct QuicServerMsIncoming {
	stream: QuicServerMs,
	subrequest_index: Vu30,
}

pub struct QuicServerMs {
	world_sender: SenderTask<WorldMessage>,
	misc_sender: SenderTask<MiscMessage>,
	receiver: ReceiverTask,
	connection: Connection,
	error: ErrorContainer,
}

impl QuicServerMsIncoming {
	pub async fn build(
		connection: Connection, world_sender_stream: BufWriter<SendStream>, world_receiver_stream: BufReader<RecvStream>, subrequest_index: Vu30,
	) -> Result<QuicServerMsIncoming, String> {
		world_sender_stream.get_ref().set_priority(1).map_err(|err| format!("failed setting priority of world stream: {err}"))?;
		let (misc_sender_stream, misc_receiver_stream) = connection.open_bi().await.map_err(|err| format!("failed creating bidirectional misc stream: {err}"))?;

		let error = ErrorContainer::default();
		let world_sender = SenderTask::<WorldMessage>::new(world_sender_stream, error.clone());
		let receiver = ReceiverTaskBuilder::new(error.clone()).add_stream::<(PlayerUpdate, u64)>(world_receiver_stream, 1 << 8).add_stream::<ClientTextInput>(BufReader::new(misc_receiver_stream), 1 << 16);
		let datagram_sender = receiver.get_sender();
		let receiver = receiver.build();

		let misc_sender = SenderTask::<MiscMessage>::new(BufWriter::new(misc_sender_stream), error.clone());

		{
			let error = error.clone();
			let connection = connection.clone();
			tokio::spawn(async move {
				while let Ok(data) = connection.read_datagram().await {
					match PlayerBulkUpdate::deserialise(&data) {
						Ok(bulk_update) => _ = datagram_sender.send(Message::PlayerBulkUpdate(bulk_update)).await,
						Err(err) => {
							error.insert_if_empty(MsError::Other(format!("failed deserialising bulk update: {err}")));
							return;
						},
					}
				}
			});
		}

		let stream = QuicServerMs { world_sender, misc_sender, receiver, connection, error };
		Ok(QuicServerMsIncoming { stream, subrequest_index })
	}

	fn send_serp_response(&self, response: Response) -> Result<(), String> {
		self.stream.world_sender.send_serp_response(response, &self.stream.error, &self.stream.connection).map_err(|err| err.to_string())
	}

	fn flush_serp_response(&self) {
		let _ = self.stream.world_sender.flush(&self.stream.error, &self.stream.connection);
	}
}

impl IncomingConnection for QuicServerMsIncoming {
	fn accept(self: Box<Self>) -> Result<Box<DynMs>, (String, String)> {
		self.send_serp_response(Response::Success(self.subrequest_index)).map_err(|err| (err, self.stream.connection.remote_address().to_string()))?;
		// No point flushing it as the server's response is only useful for the client when they get the initial Messages
		Ok(Box::new(self.stream))
	}

	fn reject(self: Box<Self>, reason: String) {
		let _ = self.send_serp_response(Response::Understood(self.subrequest_index, reason));
		self.flush_serp_response();
	}
}

impl MessageStream for QuicServerMs {
	fn send(&mut self, msg: Message) -> Result<(), MsError> {
		match msg {
			Message::World(msg) => self.world_sender.send(msg, &self.error, &self.connection),
			Message::Misc(msg) => self.misc_sender.send(msg, &self.error, &self.connection),
			_ => unimplemented!(),
		}
	}

	fn receive(&mut self) -> Result<Option<Message>, MsError> {
		match self.error.get() { // Manually need to check when receiving as receiving only fails when both streams fail, so one can fail and that won't be reported
			Some(error) => Err(error),
			None => self.receiver.receive(&self.error, &self.connection),
		}
	}

	fn flush(&mut self) -> Result<(), MsError> {
		self.world_sender.flush(&self.error, &self.connection)?;
		self.misc_sender.flush(&self.error, &self.connection)
	}

	#[cfg(feature = "client")]
	fn is_networked(&self) -> bool {
		true
	}

	fn close(self: Box<Self>) {
		self.connection.close(SHUTTING_DOWN_STATUS, SHUTTING_DOWN_MSG);
	}

	fn source(&self) -> String {
		/*
		 * This can change over time if the client ever changes networks (or NAT
		 * stuff) but it most likely will be the same as the reported address in
		 * the logs.
		 */
		self.connection.remote_address().to_canonical().to_string()
	}

	#[cfg(feature = "client")]
	fn ping(&self) -> Duration {
		self.connection.rtt()
	}

	#[cfg(feature = "client")]
	fn security(&self) -> Security {
		log::warn!("unimplemented");
		Security::Insecure
	}
}

impl MessageSer for WorldMessage {}
impl MessageSer for MiscMessage {}

impl MessageDe for (PlayerUpdate, u64) {
	fn into_message(self) -> Message {
		Message::PlayerUpdate(self.0, self.1)
	}
}

impl MessageDe for ClientTextInput {
	fn into_message(self) -> Message {
		Message::ClientTextInput(self)
	}
}
