// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::time::Instant;

use tokio::io::{AsyncWriteExt, BufWriter};
use tokio::sync::mpsc::{self, Sender, Receiver, error::TrySendError};
use quinn::{SendStream, Connection};
use bincode::Options;

use super::{CHANNEL_SIZE, MessageSer, ErrorContainer};
use super::super::MsError;

use crate::net::serp::{Response, vu30::AsyncWriteVu30};

enum SenderMessage<M> {
	/** Pushes a message to the sender task. */
	Message(M),

	/** Pushes a SERP response to the sender task. */
	SerpResponse(Response),

	/** Tells the sender task to flush the output stream. */
	Flush,
}

pub struct SenderTask<M>(Sender<(SenderMessage<M>, Instant)>);

impl<M> SenderTask<M> where M: MessageSer {
	pub fn new(mut stream: BufWriter<SendStream>, error: ErrorContainer) -> SenderTask<M> {
		let (sender, mut receiver) = mpsc::channel(CHANNEL_SIZE);

		tokio::spawn(async move {
			let mut buf = Vec::new();
			loop {
				if let Err(err) = Self::send_msg(&mut receiver, &mut stream, &mut buf).await {
					error.insert_if_empty(err);
					return;
				}
			}
		});

		SenderTask(sender)
	}

	pub fn send(&self, msg: M, error: &ErrorContainer, connection: &Connection) -> Result<(), MsError> {
		self.send_to_sender_task(SenderMessage::Message(msg), error, connection)
	}

	pub fn send_serp_response(&self, response: Response, error: &ErrorContainer, connection: &Connection) -> Result<(), MsError> {
		self.send_to_sender_task(SenderMessage::SerpResponse(response), error, connection)
	}

	pub fn flush(&self, error: &ErrorContainer, connection: &Connection) -> Result<(), MsError> {
		self.send_to_sender_task(SenderMessage::Flush, error, connection)
	}

	fn send_to_sender_task(&self, msg: SenderMessage<M>, error: &ErrorContainer, connection: &Connection) -> Result<(), MsError> {
		self.0.try_send((msg, Instant::now())).map_err(|err| match err {
			TrySendError::Closed(_) => error.get_always(connection),
			TrySendError::Full(_) => MsError::Other(String::from("sync sender is full")),
		})
	}

	async fn send_msg(receiver: &mut Receiver<(SenderMessage<M>, Instant)>, stream: &mut BufWriter<SendStream>, buf: &mut Vec<u8>) -> Result<(), MsError> {
		let (msg, send_time) = receiver.recv().await.ok_or(MsError::Disconnected(None))?;
		log::trace!(target: "perf", "task_send,{}", send_time.elapsed().as_secs_f64());
		let mut start = Instant::now();

		match msg {
			SenderMessage::Message(msg) => {
				buf.clear();
				super::bincode_options().serialize_into(&mut *buf, &msg).map_err(|err| MsError::Other(format!("cannot serialise message: {}", super::handle_bincode_error(err))))?;

				log::trace!(target: "perf", "ser,{}", start.elapsed().as_secs_f64());

				start = Instant::now();

				stream.write_bytes(buf).await.map_err(|_| MsError::Disconnected(None))?;
				log::trace!(target: "perf", "write,{}", start.elapsed().as_secs_f64());
			},
			SenderMessage::SerpResponse(response) => response.reply(stream).await.map_err(|_| MsError::Disconnected(None))?,
			SenderMessage::Flush => {
				stream.flush().await.map_err(|_| MsError::Disconnected(None))?;
				log::trace!(target: "perf", "flush,{}", start.elapsed().as_secs_f64());
			},
		}

		Ok(())
	}
}
