// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::time::Instant;

use tokio::io::BufReader;
use tokio::sync::mpsc::{self, Sender, Receiver, error::TryRecvError};
use bincode::Options;
use quinn::{RecvStream, Connection};

use super::{CHANNEL_SIZE, ErrorContainer, MessageDe};
use super::super::MsError;

use crate::net::{message::Message, serp::vu30::AsyncReadVu30};

pub struct ReceiverTaskBuilder {
	sender: Sender<Message>,
	receiver: Receiver<Message>,
	error: ErrorContainer,
}

impl ReceiverTaskBuilder {
	pub fn new(error: ErrorContainer) -> ReceiverTaskBuilder {
		let (sender, receiver) = mpsc::channel(CHANNEL_SIZE);
		ReceiverTaskBuilder { sender, receiver, error }
	}

	pub fn add_stream<M>(self, mut stream: BufReader<RecvStream>, size_limit: usize) -> Self where M: MessageDe {
		let sender = self.sender.clone();
		let error = self.error.clone();
		tokio::spawn(async move {
			let mut buf = Vec::new();
			loop {
				if let Err(err) = ReceiverTask::receive_msg::<M>(&sender, &mut stream, &mut buf, size_limit).await {
					error.insert_if_empty(err);
					return;
				}
			}
		});

		self
	}

	/**
	 * Receives a message right now from the currently added streams.
	 */
	#[cfg(feature = "client")]
	pub async fn receive(&mut self, error: &ErrorContainer, connection: &Connection) -> Result<Message, MsError> {
		self.receiver.recv().await.ok_or_else(|| error.get_always(connection))
	}

	pub fn get_sender(&self) -> Sender<Message> {
		self.sender.clone()
	}

	pub fn build(self) -> ReceiverTask {
		ReceiverTask(self.receiver)
	}
}

pub struct ReceiverTask(Receiver<Message>);

impl ReceiverTask {
	pub fn receive(&mut self, error: &ErrorContainer, connection: &Connection) -> Result<Option<Message>, MsError> {
		match self.0.try_recv() {
			Ok(msg) => Ok(Some(msg)),
			Err(TryRecvError::Empty) => Ok(None),
			Err(TryRecvError::Disconnected) => Err(error.get_always(connection)),
		}
	}

	async fn receive_msg<M>(sender: &Sender<Message>, stream: &mut BufReader<RecvStream>, buf: &mut Vec<u8>, size_limit: usize) -> Result<(), MsError> where M: MessageDe {
		buf.clear();
		stream.read_into_bytes(buf, size_limit).await.map_err(|_| MsError::Disconnected(None))?;

		let start = Instant::now();
		let msg: M = super::bincode_options().with_limit(size_limit as u64).deserialize(buf)
			.map_err(|err| MsError::Other(format!("cannot deserialise message: {}", super::handle_bincode_error(err))))?;
		log::trace!(target: "perf", "de,{}", start.elapsed().as_secs_f64());

		let _ = sender.send(msg.into_message()).await;
		Ok(())
	}
}
