// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::sync::{Arc, Mutex};

use quinn::{Connection, ConnectionError};

use super::super::MsError;

#[derive(Default, Clone)]
pub struct ErrorContainer(Arc<Mutex<Option<MsError>>>);

impl ErrorContainer {
	pub fn insert_if_empty(&self, error: MsError) {
		let Ok(mut err) = self.0.lock() else { return; };
		*err = Some(error);
	}

	pub fn get(&self) -> Option<MsError> {
		// Shouldn't block for long
		self.0.lock().map_or_else(
			|_| Some(MsError::Other(String::from("task panicked while setting error (shouldn't happen)"))),
			|lock| (*lock).clone(),
		)
	}

	pub fn get_always(&self, connection: &Connection) -> MsError {
		match self.get() {
			Some(MsError::Other(err)) => MsError::Other(err),
			Some(MsError::Disconnected(Some(reason))) => MsError::Disconnected(Some(reason)),
			Some(MsError::Disconnected(None)) | None => {
				match connection.close_reason() {
					Some(ConnectionError::TransportError(err)) => MsError::Other(format!("peer violated QUIC protocol: {err}")),
					Some(ConnectionError::ConnectionClosed(err)) => MsError::Other(format!("connection closed: {err}")),
					Some(ConnectionError::TimedOut) => MsError::Disconnected(Some(String::from("connection timed out"))),
					Some(ConnectionError::ApplicationClosed(reason)) => MsError::Disconnected(Some(format!("server closed, reason provided: {reason}"))),
					Some(err) => MsError::Other(format!("other QUIC error: {err}")),
					None => MsError::Disconnected(None),
				}
			}
		}
	}
}

