// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::time::Duration;

use quinn::{Connection, SendDatagramError};
use tokio::io::{BufReader, BufWriter};

use super::{MessageStream, MsError, Security, quic_common::{SenderTask, ReceiverTask, ReceiverTaskBuilder, ErrorContainer, MessageSer, MessageDe}};

use crate::net::{serp::{Request, PlayRequest}, message::{Message, WorldMessage, MiscMessage}, text_io::ClientTextInput};
use crate::world::player::update::PlayerUpdate;

pub struct QuicClientMs {
	world_sender: SenderTask<(PlayerUpdate, u64)>,
	misc_sender: SenderTask<ClientTextInput>,
	receiver: ReceiverTask,
	connection: Connection,
	error: ErrorContainer,
	security: Security,
}

impl QuicClientMs {
	pub async fn build(connection: Connection, request: PlayRequest, security: Security) -> Result<(QuicClientMs, Message), String> {
		let (world_sender_stream, world_receiver_stream) = Request::Play(request).send(&connection).await.map_err(|err| format!("failed making request: {err}"))?;
		let (misc_sender_stream, misc_receiver_stream) = connection.accept_bi().await.map_err(|err| format!("failed accepting bidirectional misc stream: {err}"))?;

		let error = ErrorContainer::default();
		let world_sender = SenderTask::new(world_sender_stream, error.clone());
		let misc_sender = SenderTask::new(BufWriter::new(misc_sender_stream), error.clone());
		let mut receiver_builder = ReceiverTaskBuilder::new(error.clone()).add_stream::<WorldMessage>(world_receiver_stream, 1 << 24);
		let init_msg = receiver_builder.receive(&error, &connection).await.map_err(|err| format!("failed receiving initial message: {err}"))?;
		let receiver = receiver_builder.add_stream::<MiscMessage>(BufReader::new(misc_receiver_stream), 1 << 24).build();

		Ok((QuicClientMs { world_sender, misc_sender, receiver, connection, error, security }, init_msg))
	}
}

impl MessageStream for QuicClientMs {
	fn send(&mut self, msg: Message) -> Result<(), MsError> {
		match msg {
			Message::PlayerUpdate(update, count) => self.world_sender.send((update, count), &self.error, &self.connection),
			Message::PlayerBulkUpdate(bulk_update) => {
				let Some(limit) = self.connection.max_datagram_size() else { return Ok(()); };
				let Some(data) = bulk_update.serialise(limit) else { return Ok(()); };

				debug_assert!(data.len() < limit);

				match self.connection.send_datagram(data) {
					Ok(()) | Err(SendDatagramError::UnsupportedByPeer | SendDatagramError::Disabled) => Ok(()), // Failure for some reason isn't an error
					Err(SendDatagramError::TooLarge) => { // Might happen with a TOCTOU so despite the size being limited, I shouldn't consider that an error
						log::warn!("failed sending bulk update datagram, too large");
						Ok(())
					},
					Err(SendDatagramError::ConnectionLost(_)) => Err(MsError::Disconnected(None)),
				}
			},
			Message::ClientTextInput(msg) => self.misc_sender.send(msg, &self.error, &self.connection),
			_ => unimplemented!(),
		}
	}

	fn receive(&mut self) -> Result<Option<Message>, MsError> {
		self.receiver.receive(&self.error, &self.connection)
	}

	fn flush(&mut self) -> Result<(), MsError> {
		self.world_sender.flush(&self.error, &self.connection)?;
		self.misc_sender.flush(&self.error, &self.connection)
	}

	fn is_networked(&self) -> bool {
		true
	}

	fn ping(&self) -> Duration {
		self.connection.rtt()
	}

	fn security(&self) -> Security {
		self.security
	}
}

impl MessageSer for (PlayerUpdate, u64) {}
impl MessageSer for ClientTextInput {}

impl MessageDe for WorldMessage {
	fn into_message(self) -> Message {
		Message::World(self)
	}
}

impl MessageDe for MiscMessage {
	fn into_message(self) -> Message {
		Message::Misc(self)
	}
}
