// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use serde::{Serialize, Deserialize};

use super::{scores::{Scores, DScores}, text_io::{ClientTextInput, ServerTextOutput}, timer::Timer};

use crate::world::{World, update::WorldUpdate, player::{PlayerId, update::{PlayerUpdate, PlayerBulkUpdate}}};
use crate::blocks::compressed::CompressedBlocks;

#[derive(Clone)]
pub enum Message {
	World(WorldMessage),
	Misc(MiscMessage),
	PlayerUpdate(PlayerUpdate, u64), // Client → Server, frequently sent updates of the player's input with an associated update count (to compress multiple messages)
	PlayerBulkUpdate(PlayerBulkUpdate), // Client → Server, optionally sent out as redundant player update information
	ClientTextInput(ClientTextInput), // Client → Server, either a chat message or command
}

#[derive(Clone, Serialize, Deserialize)]
pub enum WorldMessage {
	InitWorld { // Server → Client, when the client joins a game
		player_id: PlayerId, // ID of the player to be joined
		world: World,
		blocks: CompressedBlocks,
		extension: Box<[u8]>,
	},
	WorldUpdate(WorldUpdate, u64), // Server → Client, sent for every update to the world's physics and includes an acknowledgement of how many updates the server has received
	SanityCheck(World), // Server → Client, used for the client to check if the server's world matches theirs

	/*
	 * These messages are world messages so that they are sent on the same QUIC
	 * stream as the "actual" world messages.
	 *
	 * This is important for the timer so that can be in sync with the world
	 * updates, and is important for the scores to satisfy the invariant that if
	 * a player id is in the scores, that id is also in the world.
	 */
	Timer(Option<Timer>), // Server → Client, updates about the timer to be displayed
	InitScores(Scores), // Server → Client, the initial scores of the game
	DScores(DScores), // Server → Client, changes in scores
	GameStateTransition(GameStateTransition), // Server → Client, indicates how the game state changed (used by the client to play a sound)
	Extension(Box<[u8]>), // Server → Client, to allow for non-breaking extensions to the protocol without updating the SERP version
}

#[derive(Clone, Serialize, Deserialize)]
pub enum GameStateTransition {
	Start,
	End,
	Win,
}

#[derive(Clone, Serialize, Deserialize)]
pub enum MiscMessage {
	ServerTextOutput(ServerTextOutput), // Server → Client, a variety of chat-related messages
	SpectatorCount(usize), // Server → Client, reports to the users the number of spectators
	Extension(Box<[u8]>), // Server → Client, to allow for non-breaking extensions to the protocol without updating the SERP version
}
