// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::io::Read;
use std::net::{SocketAddr, IpAddr};
use std::sync::Arc;

use sysinfo::Networks;
use tokio::net::UdpSocket;

use super::{PROTOCOL_VERSION, DISCOVER_MAGIC_NUMBER, SERVER_PORT, LanServerInfo, security::SecurityPolicy};

use crate::net::{serp::vu30::ReadVu30, utils::{Ipv4Network, udp}};

pub struct LanDiscoveryServerRecvBuf(Box<[u8]>);

impl LanDiscoveryServerRecvBuf {
	pub fn new() -> LanDiscoveryServerRecvBuf {
		LanDiscoveryServerRecvBuf(vec![0; DISCOVER_MAGIC_NUMBER.len() + 5].into_boxed_slice())
	}
}

pub struct LanDiscoveryServer(Arc<UdpSocket>);

impl LanDiscoveryServer {
	pub fn build(game_addr: IpAddr) -> Result<LanDiscoveryServer, String> {
		let socket = udp::new_tokio(SocketAddr::new(LanDiscoveryServer::game_to_broadcast_addr(game_addr)?, SERVER_PORT)).map_err(|err| format!("cannot bind to port {SERVER_PORT}: {err}"))?;
		Ok(LanDiscoveryServer(Arc::new(socket)))
	}

	pub async fn receive(&self, recv_buf: &mut LanDiscoveryServerRecvBuf, policy: &mut SecurityPolicy) -> Result<SocketAddr, String> {
		loop {
			let (len, addr) = self.0.recv_from(&mut recv_buf.0).await.map_err(|err| format!("failed receiving from socket: {err}"))?;
			if !policy.allowed(addr.ip()) {
				log::info!("UDP datagram from {addr} dropped as the IP address isn't allowed");
				continue;
			}

			let mut reader = &recv_buf.0[..len];
			let mut magic = [0; DISCOVER_MAGIC_NUMBER.len()];
			if reader.read_exact(&mut magic).is_err() || magic != DISCOVER_MAGIC_NUMBER {
				log::debug!("magic number from {addr} doesn't match, got {:?}", &magic[..magic.len().min(len)]);
				continue;
			}

			let Ok(version) = reader.read_vu30() else {
				log::debug!("failed reading protocol version from {addr}");
				continue;
			};

			// Trailing data might be allowed in later versions of the protocol
			if !reader.is_empty() && version <= PROTOCOL_VERSION {
				log::debug!("trailing data found in datagram from {addr}");
				continue;
			}

			return Ok(addr);
		}
	}

	pub async fn respond(&self, info: LanServerInfo, addr: SocketAddr) -> Result<(), String> {
		self.0.send_to(&info.serialise(), addr).await.map_err(|err| format!("failed sending LAN discovery response to {addr}: {err}"))?;
		Ok(())
	}

	/**
	 * Converts the IP address the game server is bound to into a suitable IP
	 * address that the LAN discovery server should bind to. This suitable IP
	 * address is either a broadcast address or unspecified.
	 *
	 * To see why the unspecified address cannot be bound to, consider the case
	 * that a game server is started and bound to 127.0.0.1. Then if a user on
	 * another host on the local network uses LAN discovery to find games on the
	 * local network, then this game will show up but the user won't be able to
	 * access it.
	 *
	 * The solution is to bind the LAN discovery server to the directed broadcast
	 * address if binding to a specific address, or leave it unchanged if the
	 * game server bound to an unspecified address.
	 *
	 * Note that directed broadcast isn't available on IPv6. This means that LAN
	 * discovery will only work on IPv6 if the game is bound to the unspecified
	 * address, which is `::`.
	 */
	fn game_to_broadcast_addr(game_addr: IpAddr) -> Result<IpAddr, String> {
		if game_addr.is_unspecified() {
			Ok(game_addr)
		} else if let IpAddr::V4(addr) = game_addr {
			Ipv4Network::networks(&Networks::new_with_refreshed_list())
				.filter(|ip_net| ip_net.contains(addr))
				.max_by_key(|ip_net| ip_net.net_prefix)
				.map(|ip_net| IpAddr::V4(ip_net.directed_broadcast()))
				.ok_or_else(|| format!("IP address {addr} doesn't belong to any network"))
		} else {
			Err(String::from("LAN discovery server isn't supported when binding the game server to an IPv6 address that isn't `::`"))
		}
	}
}

impl Clone for LanDiscoveryServer {
	fn clone(&self) -> Self {
		LanDiscoveryServer(Arc::clone(&self.0))
	}
}
