// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::net::IpAddr;

use tokio::task;
use sysinfo::Networks;

use crate::LanDiscoveryConfig;
use crate::net::utils::NotPublic;

pub struct SecurityPolicy(Option<LocalOnly>);

struct LocalOnly(Networks);

impl SecurityPolicy {
	pub fn from_config(config: LanDiscoveryConfig) -> SecurityPolicy {
		match config {
			LanDiscoveryConfig::Disabled => unimplemented!(), // Shouldn't be called
			LanDiscoveryConfig::Local => SecurityPolicy(Some(LocalOnly(task::block_in_place(Networks::new_with_refreshed_list)))),
			LanDiscoveryConfig::Global => SecurityPolicy(None),
		}
	}

	pub fn allowed(&mut self, addr: IpAddr) -> bool {
		self.0.as_mut().is_none_or(|local_only| local_only.ip_addr_not_public(addr))
	}
}

impl NotPublic for LocalOnly {
	fn get_networks(&mut self) -> &Networks {
		&self.0
	}
}
