// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::net::{UdpSocket, SocketAddr, IpAddr, Ipv4Addr, Ipv6Addr, SocketAddrV4, SocketAddrV6};
use std::time::Instant;
use std::io::{Read, Error as IoError, ErrorKind};
use std::num::NonZeroU16;

use sysinfo::Networks;

use super::{PROTOCOL_VERSION, DISCOVER_MAGIC_NUMBER, RESPONSE_MAGIC_NUMBER, SERVER_PORT, CLIENT_PORT, MAX_LEN, LanServerInfo};

use crate::net::{serp::vu30::{ReadVu30, ExtendVu30}, utils::{Ipv4Network, udp}};

pub struct LanDiscoveryClient {
	socket: Option<Socket>,
	buf: Box<[u8]>,
}

struct Socket {
	inner: UdpSocket,
	start: Instant,
	next_time: f32,
	socket_addrs: Vec<SocketAddr>,
}

const WAIT_TIME: f32 = 5.0;
const SEND_INTERVAL: f32 = 0.5;

impl Socket {
	fn build() -> Result<Socket, String> {
		let inner = udp::new(SocketAddr::new(IpAddr::V6(Ipv6Addr::UNSPECIFIED), CLIENT_PORT)).or_else(|err| {
			// More helpful warning in the most likely reason why this failed
			let reason = if err.kind() == ErrorKind::AddrInUse {
				String::from(" because that port is already used (you might be scanning the LAN on multiple games at the same time)")
			} else {
				format!(": {err}")
			};

			log::warn!("failed binding LAN discovery to port {CLIENT_PORT}{reason}");
			log::warn!("this could mean you will get no results if a firewall is blocking incoming traffic but allowing UDP port {CLIENT_PORT}");
			udp::new(SocketAddr::new(IpAddr::V6(Ipv6Addr::UNSPECIFIED), 0))
		}).map_err(|err| format!("cannot bind socket: {err}"))?;

		inner.set_broadcast(true).map_err(|err| format!("set_broadcast failed: {err}"))?;
		inner.set_nonblocking(true).map_err(|err| format!("set_nonblocking failed: {err}"))?;

		let mut socket = Socket { inner, start: Instant::now(), next_time: SEND_INTERVAL, socket_addrs: Vec::new() };
		socket.set_addrs();
		Ok(socket)
	}

	fn update(&mut self) -> bool {
		let time = self.start.elapsed().as_secs_f32();

		if time >= self.next_time {
			self.send();
			self.next_time += SEND_INTERVAL;
			if self.next_time >= WAIT_TIME - 1.0 { self.next_time = f32::INFINITY; } // Unlikely to receive any responses
		}

		time < WAIT_TIME
	}

	fn send(&self) {
		log::debug!("Sending LAN discovery message");
		let mut msg = Vec::with_capacity(DISCOVER_MAGIC_NUMBER.len() + PROTOCOL_VERSION.len());
		msg.extend_from_slice(DISCOVER_MAGIC_NUMBER);
		msg.extend_vu30(PROTOCOL_VERSION);
		for socket_addr in &self.socket_addrs {
			/*
			 * These errors are normal as you can get them from attempting to send
			 * a message to ff02::1 on an invalid scope id. I don't have any
			 * problem with these as there isn't any way to derive the scope id
			 * from the interface name.
			 */
			if let Err(err) = self.inner.send_to(&msg, socket_addr) {
				log::debug!("cannot send to {socket_addr}: {err}");
			}
		}
	}

	fn reset(&mut self) {
		self.start = Instant::now();
		self.next_time = SEND_INTERVAL;
		self.set_addrs();
	}

	fn set_addrs(&mut self) {
		self.socket_addrs.clear();

		/*
		 * Adds the IPv4 limited broadcast address in case a router is configured
		 * to drop directed broadcasts, and the two nodes are on the same L2
		 * segment.
		 */
		self.socket_addrs.push(SocketAddr::V4(SocketAddrV4::new(Ipv4Addr::new(255, 255, 255, 255), SERVER_PORT)));

		let networks = Networks::new_with_refreshed_list();

		// IPv4 directed broadcast addresses of each IPv4 network
		for ip_net in Ipv4Network::networks(&networks) {
			self.socket_addrs.push(SocketAddr::V4(SocketAddrV4::new(ip_net.directed_broadcast(), SERVER_PORT)));
		}

		/*
		 * Many IPv6 link-local multicast addresses to all hosts.
		 *
		 * Unlike IPv4, IPv6 socket addresses also need flow information and a
		 * scope id. The flow information isn't too important and can be set to
		 * zero, but the scope id identifies the interface that should be used.
		 *
		 * Annoyingly, the `sysinfo` crate doesn't provide a way to get this scope
		 * id of an interface, but on Linux I worked out that there is one for
		 * each interface and the first id is 1.
		 */
		for scope_id in 0..=networks.len() as u32 {
			self.socket_addrs.push(SocketAddr::V6(SocketAddrV6::new(Ipv6Addr::new(0xff02, 0, 0, 0, 0, 0, 0, 1), SERVER_PORT, 0, scope_id)));
		}
	}
}

impl LanDiscoveryClient {
	pub fn new() -> LanDiscoveryClient {
		LanDiscoveryClient { socket: None, buf: vec![0; MAX_LEN + 1].into_boxed_slice() }
	}

	pub fn discover(&mut self) -> Result<(), String> {
		let socket = match &mut self.socket {
			Some(socket) => {
				socket.reset();
				socket
			}
			None => self.socket.insert(Socket::build()?),
		};

		socket.send();
		Ok(())
	}

	pub fn get(&mut self) -> Option<(LanServerInfo, SocketAddr)> {
		match self.socket.as_ref()?.inner.recv_from(self.buf.as_mut()) {
			Ok((len, src_addr)) => match LanDiscoveryClient::decode_info(&self.buf[..len]) {
				Ok(info) => {
					let socket_addr = match src_addr {
						SocketAddr::V4(addr) => SocketAddr::V4(SocketAddrV4::new(*addr.ip(), info.port)),
						SocketAddr::V6(addr) => SocketAddr::V6(SocketAddrV6::new(*addr.ip(), info.port, 0, addr.scope_id())),
					};

					Some((info, socket_addr))
				}
				Err(err) => {
					log::debug!("bad LAN discovery response from {src_addr}: {err}");
					None
				}
			},
			Err(err) if err.kind() == ErrorKind::WouldBlock => None,
			Err(err) => {
				log::warn!("failed receiving from socket: {err}");
				self.socket = None;
				None
			}
		}
	}

	pub fn update(&mut self) {
		if let Some(socket) = &mut self.socket {
			if !socket.update() {
				self.socket = None;
			}
		}
	}

	pub fn stop(&mut self) {
		self.socket = None;
	}

	pub fn open(&self) -> bool {
		self.socket.is_some()
	}

	fn decode_info(buf: &[u8]) -> Result<LanServerInfo, String> {
		if buf.len() > MAX_LEN {
			return Err(format!("received more than {MAX_LEN} bytes"));
		}

		let mut reader = buf;
		let mut magic = [0u8; RESPONSE_MAGIC_NUMBER.len()];
		reader.read_exact(&mut magic).map_err(Self::too_few_bytes)?;
		if magic != RESPONSE_MAGIC_NUMBER {
			return Err(format!("magic number doesn't match, got {magic:?}"));
		}

		let version = reader.read_vu30().map_err(Self::too_few_bytes)?;
		if version != PROTOCOL_VERSION {
			return Err(format!("protocol version doesn't match, expected {PROTOCOL_VERSION}, got {version}"));
		}

		let request_type = reader.read_vu30().map_err(Self::too_few_bytes)?;
		let request_version = reader.read_vu30().map_err(Self::too_few_bytes)?;

		let mut get_u16 = || -> Result<u16, String> {
			let mut two_bytes = [0u8; 2];
			reader.read_exact(&mut two_bytes).map_err(Self::too_few_bytes)?;
			Ok(u16::from_le_bytes(two_bytes))
		};

		let (port, cur_players, max_players) = (get_u16()?, get_u16()?, NonZeroU16::new(get_u16()?));

		let mut strings = [Vec::new(), Vec::new(), Vec::new()];
		let mut string_index = 0;

		for x in reader.bytes() {
			// I think this is infallible but I don't want to add an unwrap
			let Ok(byte) = x else { return Err(String::from("failed reading bytes")); };

			if byte == 0 {
				string_index = (string_index + 1).min(strings.len() - 1);
			} else {
				strings[string_index].push(byte);
			}
		}

		let [game_id, name, desc] = strings;

		let game_id = String::from_utf8(game_id).map_err(|err| format!("invalid UTF-8 in game id: {err}"))?.into_boxed_str();
		let name = String::from_utf8(name).map_err(|err| format!("invalid UTF-8 in name: {err}"))?.into_boxed_str();
		let desc = String::from_utf8(desc).map_err(|err| format!("invalid UTF-8 in description: {err}"))?.into_boxed_str();

		Ok(LanServerInfo { version, request_type, request_version, port, cur_players, max_players, game_id, name, desc })
	}

	fn too_few_bytes(_err: IoError) -> String { String::from("too few bytes") }
}
